#include "mbed.h"
#include "stm32746g_discovery_lcd.h"

#include "Interface.h"

DigitalOut                      led(LED1);
Serial                          serial(PA_9,PB_7);          // serial 1
AnalogIn                        analogIn1(A0);
AnalogIn                        analogIn2(A1);
VALUES_STRUCT                   buffer[BUFFER_SIZE];
uint8_t                         commandBuffer[100];
uint32_t                        commandLength;

void sendBuffer(uint8_t *buffer,uint32_t length)
{
    while (length>0)
    {
        serial.putc(*buffer);

        buffer++;
        length--;
    }
}

void sendCommand(HEADER_TYPE_ENUM type,uint8_t *buffer,uint32_t length)
{
    HEADER_STRUCT               header;

    header.magic=HEADER_MAGIC;
    header.type=type;
    header.length=length;

    sendBuffer((uint8_t *)&header,sizeof(header));
    sendBuffer(buffer,length);
}

bool readWithTimeout(uint8_t *buffer)
{
    uint32_t counter;

    counter=0;
    while (counter<10000)
    {
        if (serial.readable())
        {
            (*buffer)=serial.getc();
            return true;
        }

        wait_us(50);

        counter++;
    }

    // re-init serial port
    serial.baud(115200);

    return false;
}

HEADER_TYPE_ENUM readCommand(void)
{
    HEADER_STRUCT               *header;
    uint32_t                    index;
    HEADER_TYPE_ENUM            type;

    for (index=0;index<sizeof(HEADER_STRUCT);index++)
    {
        if (readWithTimeout(&commandBuffer[index])==false)
            return HT_NONE;
    }

    header=(HEADER_STRUCT *)commandBuffer;
    if (header->magic!=HEADER_MAGIC)
            return HT_NONE;

    type=(HEADER_TYPE_ENUM)header->type;
        
    if (header->length>0)
    {
        commandLength=header->length;

        for (index=0;index<commandLength;index++)
        {
            if (readWithTimeout(&commandBuffer[index])==false)
            return HT_NONE;
        }
    }

    return type;
}

int main()
{
    int32_t                         index;
    int32_t                         indexPrevious;
    int32_t                         stop;
    HEADER_TYPE_ENUM                type;
    bool                            capture;
    float                           triggerLevel;
    uint8_t                         triggerMode;
    uint16_t                        delay;
    bool                            triggered;

    serial.baud(115200);

    BSP_LCD_Init();
    BSP_LCD_DisplayOff();

    capture=false;
    triggerLevel=0.0;
    triggerMode=TM_NONE;
    delay=0;

    for (;;)
    {
        if (serial.readable())
        {
            type=readCommand();
            switch (type)
            {
                case HT_INFO_REQUEST:
                    {
                        INTERFACE_INFO_RESPONSE_STRUCT              InfoResponse;
                        
                        memset(&InfoResponse,0,sizeof(InfoResponse));
                        InfoResponse.versionMajor=1;
                        InfoResponse.versionMinor=0;
                        snprintf(InfoResponse.name,sizeof(InfoResponse.name),"Scope");
                        sendCommand(HT_INFO_RESPONSE,(uint8_t *)&InfoResponse,sizeof(InfoResponse));
                    }
                    break;
                    
                case HT_CAPTURE_START:
                    {
                        INTERFACE_CAPTURE_START_STRUCT              *captuerStart;
                        
                        if (commandLength==sizeof(INTERFACE_CAPTURE_START_STRUCT))
                        {
                            captuerStart=(INTERFACE_CAPTURE_START_STRUCT *)commandBuffer;
    
                            triggerLevel=captuerStart->triggerLevel;
                            triggerMode=captuerStart->triggerMode;
                            delay=captuerStart->delay;

                            // seems our measurement loop needs 5 µs per loop
                            if (delay>5)
                                delay-=5;
    
                            capture=true;
                        }
                    }
                    break;
            }
        }
        
        if (capture==true)
        {
            led=1;
    
            // dummy read for nicer results
            for (index=0;index<5;index++)
            {
                analogIn1.read();
                analogIn2.read();
            }
     
            stop=0;
            index=0;
            indexPrevious=0;
            triggered=false;
            for (;;)
            {
                buffer[index].values[0]=analogIn1.read();
                buffer[index].values[1]=analogIn2.read();
        
                // check for trigger
                if ((triggered==false) && (index>(BUFFER_SIZE/2)))
                {
                    // we trigger only on channel 1
                    if ((triggerMode & TM_RAISING)!=0)
                        if (buffer[indexPrevious].values[0]<triggerLevel)
                            if (buffer[index].values[0]>=triggerLevel)
                                triggered=true;
                    if ((triggerMode & TM_FALLING)!=0)
                        if (buffer[indexPrevious].values[0]>triggerLevel)
                            if (buffer[index].values[0]<=triggerLevel)
                                triggered=true;

                    if (triggered==true)
                        stop=(index + (BUFFER_SIZE / 2)) % BUFFER_SIZE;
                }

                indexPrevious=index;

                index++;
                if (index>=BUFFER_SIZE)
                    index=0;

                if (index==stop)
                    break;

                wait_us(delay);
            }

            led=0;
            capture=false;

            if ((triggered==true) || ((triggerMode & TM_AUTO)!=0))
            {
                HEADER_STRUCT               header;

                header.magic=HEADER_MAGIC;
                header.type=HT_DATA;
                header.length=sizeof(buffer);

                // send header
                sendBuffer((uint8_t *)&header,sizeof(header));
                // send part 1
                sendBuffer((uint8_t *)&buffer[stop],((BUFFER_SIZE-stop)*sizeof(VALUES_STRUCT)));
                // send part 2
                if (stop>0)
                    sendBuffer((uint8_t *)&buffer[0],(stop*sizeof(VALUES_STRUCT)));
            }
            else
                sendCommand(HT_DATA,(uint8_t *)&buffer,0);
        }
    }
}
