/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"
#include "CyclingSpeedAndCadenceService.h"
#include "ble/services/BatteryService.h"
#include "ble/services/DeviceInformationService.h"

BLE  ble;
Timer timer;

const static char     DEVICE_NAME[]        = "CSC1";
static const uint16_t uuid16_list[]        = {GattService::UUID_CYCLING_SPEED_AND_CADENCE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE};

uint32_t t = 0;
uint32_t nextWheel = 0;
uint32_t nextCrank = 0;
static volatile bool  triggerWheel = false;
static volatile bool  triggerCrank = false;

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    ble.gap().startAdvertising(); // restart advertising
}


void onTick(void)
{
    ++t;
    
    if (t >= nextWheel)
    {
        triggerWheel = true;
        nextWheel += 7 + (rand() % 10);
    }
    
    if (t >= nextCrank)
    {
        triggerCrank = true;    
        nextCrank += 8 + (rand() % 20);
    }
}

int main(void)
{
    Ticker ticker;
    ticker.attach(onTick, 0.1);
    timer.start();

    ble.init();
    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service. */
    uint32_t wheelCounter = 100; // init Wheel to 100revs
    uint16_t crankCounter = 10; // init crank to 10revs
    CyclingSpeedAndCadenceService cscService(ble,
        CyclingSpeedAndCadenceService::MODE_SPEED_CADENCE,
        CyclingSpeedAndCadenceService::LOCATION_CHAINSTAY);

    /* Setup auxiliary service. */
    DeviceInformationService deviceInfo(ble, "ROB", "Model1", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");

    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::CYCLING_SPEED_AND_CADENCE_SENSOR);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms */
    ble.gap().startAdvertising();
    
    nextWheel = 7 + (rand() % 10);
    nextCrank = 8 + (rand() % 20);

    // infinite loop
    while (1) {
        // check for trigger from periodicCallback()
        if (ble.getGapState().connected)
        {
            if (triggerCrank && triggerWheel)
            {
                uint16_t when = (timer.read() * 1024);
                cscService.updateCounters(++wheelCounter, ++crankCounter, when);
                triggerWheel = false;
                triggerCrank = false;
            }
            else if (triggerWheel)
            {
                uint16_t when = (timer.read() * 1024);
                cscService.updateWheelCounter(++wheelCounter, when);
                triggerWheel = false;
            }
            else if (triggerCrank)
            {
                uint16_t when = (timer.read() * 1024);
                cscService.updateCrankCounter(++crankCounter, when);
                triggerCrank = false;
            }
            else
            {
                ble.waitForEvent(); // low power wait for event
            }
        } else {
            ble.waitForEvent(); // low power wait for event
        }
    }
}
