/*****************************************************************************
 * Type : C++
 * File : MAX72XX.cpp
 * Dec. : MAX7219 & MAX7221 Software SPI library [7 SEGMENT]
 * Copyright (c) 2013-2014, Bird Techstep, tbird_th@hotmail.com
 *
 * Remark : Original codr from LedControl Library [Arduino]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *****************************************************************************/
#include "MAX72XX.h"

#define LOW  0
#define HIGH 1

//The opcodes for the MAX7221 and MAX7219
#define OP_NOOP         0x00

#define OP_DIGIT0       0x01
#define OP_DIGIT1       0x02
#define OP_DIGIT2       0x03
#define OP_DIGIT3       0x04
#define OP_DIGIT4       0x05
#define OP_DIGIT5       0x06
#define OP_DIGIT6       0x07
#define OP_DIGIT7       0x08

#define OP_DECODEMODE   0x09    // 0x00 = No decode for digits 7–0
                                // 0x01 = Code B decode for digit 0, No decode for digits 7–1
                                // 0x0F = Code B decode for digits 3–0, No decode for digits 7–4
                                // 0xFF = Code B decode for digits 7–0
#define OP_INTENSITY    0x0A    // 1/32..31/32 [0..15]
#define OP_SCANLIMIT    0x0B    // 0x00 = Display digit  0
                                // 0x01 = Display digits 0 1
                                // 0x02 = Display digits 0 1 2
                                // 0x03 = Display digits 0 1 2 3
                                // 0x04 = Display digits 0 1 2 3 4
                                // 0x05 = Display digits 0 1 2 3 4 5
                                // 0x06 = Display digits 0 1 2 3 4 5 6
                                // 0x07 = Display digits 0 1 2 3 4 5 6 7
#define OP_SHUTDOWN     0x0C    // 0x00 = Shutdown Mode
                                // 0x01 = Normal Operation
#define OP_DISPLAYTEST  0x0F    // 0x00 = Normal Operation
                                // 0x01 = Display Test Mode


MAX72XX::MAX72XX(PinName mosi_pin, PinName miso_pin, PinName sclk_pin, PinName cs_pin, int numDevices) :cs(cs_pin) {
    mosi = new DigitalOut(mosi_pin);
    miso = new DigitalIn(miso_pin);
    sclk = new DigitalOut(sclk_pin);
    format(8);
    frequency();
    
    if(numDevices<=0 || numDevices>8 )
    numDevices=8;
    maxDevices=numDevices;
    cs = 1;
    for(int i=0;i<64;i++) 
    status[i] = 0x00;
    for(int i=0;i<maxDevices;i++) {
        spiTransfer(i,OP_DISPLAYTEST,0);    // Display test off on startup
        setScanLimit(i,7);                  // Scanlimit is set to max on startup
        spiTransfer(i,OP_DECODEMODE,0);     // Decode-mode is done in source
        clearDisplay(i);
        shutdown(i,true);                   // We go into shutdown-mode on startup
    }
}

MAX72XX::~MAX72XX()
{
    delete mosi;
    delete miso;
    delete sclk;
}

void MAX72XX::frequency(int hz)
{
    this->freq = hz;
}

void MAX72XX::format(int bits, int mode)
{
    this->bits = bits;
    this->mode = mode;
    polarity = (mode >> 1) & 1;
    phase = mode & 1;
    sclk->write(polarity);
}

int MAX72XX::write(int value)
{
    int read = 0;
    for (int bit = bits-1; bit >= 0; --bit)
    {
        mosi->write(((value >> bit) & 0x01) != 0);
 
        if (phase == 0)
        {
            if (miso->read())
                read |= (1 << bit);
        }
 
        sclk->write(!polarity);
 
        wait(1.0/freq/2);
 
        if (phase == 1)
        {
            if (miso->read())
                read |= (1 << bit);
        }
 
        sclk->write(polarity);
 
        wait(1.0/freq/2);
    }
    return read;
}

void MAX72XX::decodeMode(int addr, uint8_t mode) {
    if(addr<0 || addr>=maxDevices)
    return;
    spiTransfer(addr, OP_DECODEMODE, mode);
}

void MAX72XX::displayTest(int addr, bool status) {
    if(addr<0 || addr>=maxDevices)
    return;
    if(status)
    spiTransfer(addr, OP_DISPLAYTEST,1);
    else
    spiTransfer(addr, OP_DISPLAYTEST,0);
}

void MAX72XX::shutdown(int addr, bool status) {
    if(addr<0 || addr>=maxDevices)
    return;
    if(status)
    spiTransfer(addr, OP_SHUTDOWN,0);
    else
    spiTransfer(addr, OP_SHUTDOWN,1);
}

void MAX72XX::setScanLimit(int addr, int limit) {
    if(addr<0 || addr>=maxDevices)
    return;
    if(limit>=0 || limit<8)
    spiTransfer(addr, OP_SCANLIMIT,limit);
}

void MAX72XX::setIntensity(int addr, int intensity) {
    if(addr<0 || addr>=maxDevices)
    return;
    if(intensity>=0 || intensity<16)    
    spiTransfer(addr, OP_INTENSITY,intensity);
}

void MAX72XX::clearDigit(int addr, int digit) {
    int offset;

    if(addr<0 || addr>=maxDevices)
    return;
    //spiTransfer(addr, digit+1,0);
    offset = addr*8;
    for(int i=0;i<8;i++) {
        status[offset+i] = 0;
        spiTransfer(addr, digit+1,status[offset+i]);
    }
}

void MAX72XX::clearDisplay(int addr) {
    int offset;

    if(addr<0 || addr>=maxDevices)
    return;
    offset = addr*8;
    for(int i=0;i<8;i++) {
        status[offset+i] = 0;
        spiTransfer(addr, i+1,status[offset+i]);
    }
}

void MAX72XX::setDigit(int addr, int digit, uint8_t value, bool dp) {
    int offset;
    uint8_t v;

    if(addr<0 || addr>=maxDevices)
    return;
    if(digit<0 || digit>7 || value>15)
    return;
    offset = addr*8;
    v = charTable[value];
    if(dp)
    v |= 0x80;
    status[offset+digit]=v;
    spiTransfer(addr, digit+1,v);
}

void MAX72XX::setChar(int addr, int digit, char value, bool dp) {
    int offset;
    uint8_t index,v;

    if(addr<0 || addr>=maxDevices)
    return;
    if(digit<0 || digit>7)
    return;
    offset = addr*8;
    index  = value;
    if(index >127) {
    index = 32;         //No defined beyond index 127, so we use the space char
    }
    v = charTable[index];
    if(dp)
    v |= 0x80;
    status[offset+digit] = v;
    spiTransfer(addr, digit+1,v);
}

void MAX72XX::spiTransfer(int addr, volatile uint8_t opcode, volatile uint8_t data) {
    //Create an array with the data to shift out
    int offset   = addr*2;
    int maxbytes = maxDevices*2;

    for(int i=0;i<maxbytes;i++)
    spidata[i] = 0;
    //put our device data into the array
    spidata[offset+1] = opcode;
    spidata[offset]   = data;
    //enable the line 
    cs = 0;
    //Now shift out the data 
    for(int i=maxbytes;i>0;i--)
    this->write(spidata[i-1]);
    //latch the data onto the display
    cs = 1;
}    
