/*****************************************************************************
 * Type : Header
 * File : MAX72XX.h
 * Dec. : MAX7219 & MAX7221 Software SPI library [7 SEGMENT]
 * Copyright (c) 2013-2014, Bird Techstep, tbird_th@hotmail.com
 *
 * Remark : Original codr from LedControl Library [Arduino]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *****************************************************************************/
#ifndef MAX72XX_H
#define MAX72XX_H

#include "mbed.h"

// Segments to be switched on for characters and digits on 7-Segment Displays
const static char charTable[128] = {
    0x7E, 0x30, 0x6D, 0x79, 0x33, 0x5B, 0x5F, 0x70, // 0, 1, 2, 3, 4, 5, 6, 7,
    0x7F, 0x7B, 0x77, 0x1F, 0x0D, 0x3D, 0x4F, 0x47, // 8, 9, A, b, c, d, E, F,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //  ,  ,  ,  ,  ,  ,  ,  ,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //  ,  ,  ,  ,  ,  ,  ,  ,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //  ,  ,  ,  ,  ,  ,  ,  ,
    0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x80, 0x00, //  ,  ,  ,  , ., -, .,  ,
    0x7E, 0x30, 0x6D, 0x79, 0x33, 0x5B, 0x5F, 0x70, // 0, 1, 2, 3, 4, 5, 6, 7,
    0x7F, 0x7B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 8, 9,  ,  ,  ,  ,  ,  ,
    0x00, 0x77, 0x1F, 0x0D, 0x3D, 0x4F, 0x47, 0x00, //  , A, b, c, d, E, F,  ,
    0x37, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, // H,  ,  ,  , L,  ,  ,  ,
    0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // P,  ,  ,  ,  ,  ,  ,  ,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, //  ,  ,  ,  ,  ,  ,  , _,
    0x00, 0x77, 0x1F, 0x0D, 0x3D, 0x4F, 0x47, 0x00, //  , A, b, c, d, E, F,  ,
    0x37, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, // H,  ,  ,  , L,  ,  ,  ,
    0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // P,  ,  ,  ,  ,  ,  ,  ,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //  ,  ,  ,  ,  ,  ,  ,  ,
};

class MAX72XX {
public:
    /*************************************************************************
     * Create MAX72XX Object.
     *************************************************************************
     * @param mosi_pin   : MOSI Digital pin. [Master Out Slave In]
     * @param miso_pin   : MISO Digital pin. [Master In Slave Out]
     * @param sclk_pin   : SCLK Digital pin. [SPI Clock]
     * @param cs_pin     : CS   Digital pin. [Chip Select]
     * @param numDevices : Maximum number of devices that can be controled.
     *************************************************************************/
    MAX72XX(PinName mosi_pin, PinName miso_pin, PinName sclk_pin, PinName cs_pin, int numDevices=1);
    
    /*************************************************************************
     * Destructor.
     *************************************************************************/
    ~MAX72XX();
    
    /*************************************************************************
     * Specify SPI format.
     *************************************************************************
     * @param bits   : 8 or 16 are typical values.
     * @param mode   : 0, 1, 2, or 3 phase (bit1) and idle clock (bit0).
     *************************************************************************/
    void format(int bits, int mode = 0);
    
    /*************************************************************************
     * Specify SPI clock frequency.
     *************************************************************************
     * @param hz  : frequency (optional, defaults to 10000000).
     *************************************************************************/
    void frequency(int hz = 10000000);
    
    /*************************************************************************
     * Write data and read result.
     *************************************************************************
     * @param value  : data to write (see format for bit size).
     *                 returns value read from device.
     *************************************************************************/
    int write(int value);
    
    /*************************************************************************
     * Display a character on a 7-Segment display.
     *************************************************************************
     * Dec. : There are only a few characters that make sense here :
     *        '0','1','2','3','4','5','6','7','8','9','0',
     *        'A','b','c','d','E','F','H','L','P',
     *        '.','-','_',' '.
     * @param addr   : Number of device.
     * @param digit  : The position of the character on the display (0..7).
     * @param value  : The character to be displayed. 
     * @param dp     : Sets the decimal point.
     *************************************************************************/
    void setChar(int addr, int digit, char value, bool dp);
    
    /*************************************************************************
     * Display a hexadecimal digit on a 7-Segment Display.
     *************************************************************************
     * @param addr   : Number of device.
     * @param digit  : The position of the digit on the display (0..7)
     * @param value  : The value to be displayed. (0x00..0x0F)
     * @param dp     : Tets the decimal point.
     *************************************************************************/
    void setDigit(int addr, int digit, uint8_t value, bool dp);
    
    /*************************************************************************
     * Switch all Leds on the display off.
     *************************************************************************
     * @param addr   : Number of device.
     *************************************************************************/
    void clearDisplay(int addr);
    
    /*************************************************************************
     * Switch all Leds on the display off.
     *************************************************************************
     * @param addr  : Number of device.
     * @param digit : 0-7
     *************************************************************************/
    void clearDigit(int addr, int digit);

    /*************************************************************************
     * Set the brightness of the display.
     *************************************************************************
     * @param addr      : Number of device.
     * @param intensity : Brightness of the display. (0..15)
     *************************************************************************/
    void setIntensity(int addr, int intensity);
    
    /*************************************************************************
     * Set the number of digits to be displayed.
     *************************************************************************
     * @param addr   : Number of device.
     * @param limit  : number of digits to be displayed (1..8)
     *************************************************************************/
    void setScanLimit(int addr, int limit);
    
    /*************************************************************************
     * Set the shutdown (power saving) mode for the device.
     *************************************************************************
     * @param addr    : Number of device.
     * @param status  : If true the device goes into power-down mode. 
     *                  Set to false for normal operation
     *************************************************************************/
    void shutdown(int addr, bool status);

    /*************************************************************************
     * Display test .
     *************************************************************************
     * @param addr    : Number of device.
     * @param status  : If true test false for mormal
     *************************************************************************/
    void displayTest(int addr, bool status);
    
    /*************************************************************************
     * Decode Mode.
     *************************************************************************
     * @param addr  : Number of device.
     * @param mode  : 0x00, 0x01, 0x0F, 0xFF
     *************************************************************************/
    void decodeMode(int addr, uint8_t mode);

private: 
    /*************************************************************************
     * Send out a single command to the device.
     *************************************************************************
     * @param addr    : Number of Devices.
     * @param opcode  : Register Address Map.
     * @param data    : The value to be displayed. (0x00..0x0F)
     *************************************************************************/
    void spiTransfer(int addr, uint8_t opcode, uint8_t data);
    
    // mosi_pin, miso_pin, sclk_pin, cs_pin, numDevices
    DigitalOut* mosi;
    DigitalIn*  miso;
    DigitalOut* sclk;
    int numDevices;
    int port;
    int bits;
    int mode;
    int polarity; // idle clock value
    int phase; // 0=sample on leading (first) clock edge, 1=trailing (second)
    int freq;
    
    uint8_t spidata[16]; // The array for shifting the data to the devices.
    uint8_t status[64]; // We keep track of the led-status for all 8 devices in this array.
    int maxDevices; // The maximum number of devices we use.
    
protected:
    //SPI _spi;
    DigitalOut cs;
    
};
 
#endif // MAX72XX_H