// T. Bronez. rev. 2016-01-23
// Based on https://developer.mbed.org/users/loopsva/code/WDT4088/
// Put into Singleton design pattern for easy use throughout a program

#ifndef WATCHDOG_4088_H
#define WATCHDOG_4088_H

#include "mbed.h"

/** Routines to enable and feed the Watchdog timer for the LPC4088.
 *
 *   The LPC4088 has a fixed, internal 500KHz oscillator which is divided
 *   by 4 to give an 8uS master clock to the Watchdog countdown timer.
 *
 *   User inputs a value from 1 to 134 seconds when initializing the Watchdog
 *   timer. The user's input number is multiplied by 125k and then placed into 
 *   the countdown timer.
 *
 *   The user is responsible for feeding the Watchdog before the timeout
 *   interval expires, otherwise the LPC4088 will automatically reboot.
 *
 * @code
 * #include "mbed.h"
 * #include "Watchdog.h" 
 *
 * int main() {
 *    Watchdog* pwd = Watchdog::getInstance();
      pwd->enable(20);        // enable the watchdog with a 20 second timeout
 *    while(1) {
 *        wait_ms(1000);      // do some code
 *        pwd->feed();        // feed the watchdog before 20 seconds is up
 *     }
 * }
 * @endcode
 *
 */
 
/* Watchdog controller class
 */
class Watchdog {

    public:
    
        /* Get the singleton instance, creating it if needed
         * @returns pointer to watchdog instance
         */
        static Watchdog* getInstance();
    
        /** Enable the watchdog
         * @param int timeout in seconds. Range (1 - 134)
         */
        void enable(int WDTseconds);
        
        /** Keep watchdog quiet by feeding
         */
        void feed();
    
        /** Identify reason for reset (watchdog or normal)
         * @returns true if reset due to watchdog
         */
        static bool isWatchdogReset();
        
    private:
        // Private constructor
        Watchdog();
        // Private copy constructor w/implementation
        Watchdog(Watchdog const&);
        // Private assignment operator w/implementation
        Watchdog& operator=(Watchdog const&);
        // Global static point to singleton instance
        static Watchdog* _pInstance;
            
        int WDTseconds;
};

#endif
