// Header to represent serial interface to MTK3339 GPS chip
// Original author: Embedded Artists
// Revised by T. Bronez, 2015-07-31

#ifndef MTK3339_H
#define MTK3339_H
#include <time.h>

/** 
 * An interface to the MTK3339 GPS module.
 */
class MTK3339 {
public:

    enum NmeaSentence {
        NMEA_NONE = 0x00,
        NMEA_GGA  = 0x01,
        NMEA_GSA  = 0x02,
        NMEA_GSV  = 0x04,
        NMEA_RMC  = 0x08,
        NMEA_VTG  = 0x10
    };
       
    struct RmcType {
        char status;        // A valid, V invalid, blank unknown
        double lat_dd;      // latitude in decimal degrees
        double lon_dd;      // longitude in decimal degrees
        struct tm gps_tm;   // date and time (to seconds)
        int msec;           // millseconds
    };
    
    /** 
     * Create an interface to the MTK3339 GPS module
     *
     * @param tx UART TX line pin
     * @param rx UART RX line pin
     */
    MTK3339(PinName tx, PinName rx);
         
    /** 
     * Start to read data from the GPS module.
     * 
     * @param fptr A pointer to a void function that will be called when there 
     * is data available. 
     * @param mask specifies which sentence types (NmeaSentence) that are of 
     * interest. The callback function will only be called for messages 
     * specified in this mask.
     */     
    void start(void (*fptr)(void), int mask);

    /** 
     * Start to read data from the GPS module.
     * 
     * @param tptr pointer to the object to call the member function on
     * @param mptr pointer to the member function to be called
     * @param mask specifies which sentence types (NmeaSentence) that are of 
     * interest. The member function will only be called for messages 
     * specified in this mask.
     */        
    template<typename T>
    void start(T* tptr, void (T::*mptr)(void), int mask) {
        if((mptr != NULL) && (tptr != NULL) && mask) {
            _dataCallback.attach(tptr, mptr);
            _sentenceMask = mask;
            _serial.attach(this, &MTK3339::uartIrq, Serial::RxIrq);            
        }
    }    
    
    /** 
     * Stop to read data from GPS module
     */
    void stop();
    
    /** 
     * Get the type of the data reported in available data callback.
     * This method will only return a valid type when called within the
     * callback. 
     */    
    NmeaSentence getAvailableDataType();

    /**
     * Time, position and date related data
     */
    enum PubConstants {
        MSG_BUF_SZ = 100
    };
    // Leave room for a terminating \0 in each buffer
    char ggaMsg[MSG_BUF_SZ+1];  // longest observed strlen = 72
    char gsaMsg[MSG_BUF_SZ+1];  // longest observed strlen = 56
    char gsvMsg[MSG_BUF_SZ+1];  // longest observed strlen = 68
    char rmcMsg[MSG_BUF_SZ+1];  // longest observed strlen = 70
    char vtgMsg[MSG_BUF_SZ+1];  // longest observed strlen = 37

    RmcType rmc;
    void decodeRMC();

private:

    enum PrivConstants {
        MTK3339_BUF_SZ = 255
    };
    
    enum DataState {
        StateStart = 0,
        StateData
    };
    
    FunctionPointer _dataCallback;
    char _buf[MTK3339_BUF_SZ];
    int _bufPos;
    DataState _state;
    int _sentenceMask;
    NmeaSentence _availDataType;
    Serial _serial;

    void saveRMC(char* data, int dataLen);
    void saveGGA(char* data, int dataLen);
    void saveGSA(char* data, int dataLen);
    void saveGSV(char* data, int dataLen);
    void saveVTG(char* data, int dataLen);
    void saveData(char* data, int len);
    void uartIrq();

};

#endif

