///////////////////////////////////////////////////////////////////////////////
// Signal Generator
// Author: Chris Taylor (taylorza)
#ifndef __SIGNALGENERATOR_H__
#define __SIGNALGENERATOR_H__

/** Simplifies generation of a high frequency signal on any pin with optional support for a carrier frequency.
*/
class SignalGenerator
{
    public:
        /** Create a SignalGenerator tied to the specified pin. */
        SignalGenerator(PinName pin);
        
        /** Clean up the SignalGenerator data. */
        ~SignalGenerator();
        
        /** Set the state of the pin associated with the SignalGenerator. */  
        void set(bool pinState);
        
        /** Generates the square wave signal on the pin associated with the SignalGenerator. 
         * @param initialState Defines the initial state of the signal pin
         * @param timingBuffer Specificies the wime periods in microseconds before the signal pin changes state
         * @param bufferCount The count of transition times passed in the timingBuffer
         * @param disableInterrupts If true disables interrupts during the generation of the signal
         * @param lastStateHoldTime The time in microseconds that the last state is held
         * @param carrierFrequency The carrier frequency in Hz
        */  
        void set(
            bool initialState, 
            uint32_t timingBuffer[], 
            uint16_t bufferCount, 
            bool disableInterrupts,
            uint32_t lastStateHoldTime = 0, 
            int32_t carrierFrequency = -1);
        
        /** Asynchronously generates the square wave signal on the pin associated with the SignalGenerator. 
         * @param initialState Defines the initial state of the signal pin
         * @param timingBuffer Specificies the wime periods in microseconds before the signal pin changes state
         * @param bufferCount The count of transition times passed in the timingBuffer
         * @param repeat If true the signal generation will wrap around to the begining of the timingBuffer after each iteration through the buffer completes.
         * @note To stop the asynchronous signal you can call any of the non-async set commands to set the pin to a final state.
        */    
        void setAsync(
            bool initialState, 
            uint32_t timingBuffer[], 
            uint16_t bufferCount, 
            bool repeat);
            
    private:
        void asyncStep();
        void stopAsync();
            
    private:
        DigitalOut  _pin; 
        Timeout     *_pTicker; 
        uint32_t    *_pInternalTimingBuffer;
        uint16_t    _bufferCount;
        uint16_t    _bufferIndex;
        bool        _repeat;
};
#endif //__SIGNALGENERATOR_H__