///////////////////////////////////////////////////////////////////////////////
// Signal Capture Library
// Author: Chris Taylor (taylorza)
#ifndef __INTERRUPTPINCAPTURE_H__
#define __INTERRUPTPINCAPTURE_H__

/** Captures the digital wave form of an interrupt capable pin into a timing buffer. */
class InterruptPinCapture
{
    public:
        /** Create an instance of InterruptPinCapture connected to the specified pin */
        InterruptPinCapture(PinName pin, PinMode mode);
        
        /** Maximum time to wait for waveform data to arrive or completely fill the timing buffer. 
         * @param us Timeout specified in microseconds
         */
        void setReadTimeout(uint32_t us);
        
        /** Waits for the pin to transition to the trigger state and proceeds to capture the
         * pins transition timing into the timing buffer.
         * @param triggerState State that the pin must transistion to before the readings are captured
         * @param pReadings Pointer to the timing buffer that will contain the readings in microseconds
         * @param count The maximum number of readings that can be held in the buffer
         * @returns The number of timings captured in the buffer.
         */      
        int read(bool triggerState, uint32_t *pReadings, int count);
        
        /** Immediately start capturing the pin transition timing into the timing buffer.
         * @param pInitialState Pointer to a bool that will have the initial state of the pin at the time the capture started.
         * @param pReadings Pointer to the timing buffer that will contain the readings in microseconds
         * @param count The maximum number of readings that can be held in the buffer
         * @returns The number of timings captured in the buffer.
         */      
        int read(bool *pInitialState, uint32_t *pReadings, int count);
        
    private:
        int readInternal(bool *pPinState, uint32_t *pReadings, int count, bool waitForTrigger);
        
    private:
        void onPinTransition();
        
        enum State
        {
            Stopped,
            WaitForTrigger,
            StartCapturing,
            Capturing
        };
        
    private:
        bool _started;
        uint32_t _readTimeout;        
        volatile State _state;        
        InterruptIn _signalPin;            
        
        uint32_t *_pBuffer;
        int _bufferMaxCount;
        int _bufferIndex;
        int _startPinState;
        
        int _triggerState;
        uint32_t _lastTransitionTime;
        Timer _timer;
};
#endif //__INTERRUPTPINCAPTURE_H__