///////////////////////////////////////////////////////////////////////////////
// Signal Capture Library
// Author: Chris Taylor (taylorza)
#include "mbed.h"
#include "DigitalPinCapture.h"

DigitalPinCapture::DigitalPinCapture(PinName pin, PinMode mode) :
    _signalPin(pin, mode),
    _readTimeout(10000000)
{
    
}

void DigitalPinCapture::setReadTimeout(uint32_t us)
{
    _readTimeout = us;
}

int DigitalPinCapture::read(bool triggerState, uint32_t *pReadings, int count)
{
    return readInternal(&triggerState, pReadings, count, true);
}

int DigitalPinCapture::read(bool *pInitialState, uint32_t *pReadings, int count)
{
    return readInternal(pInitialState, pReadings, count, false);
}

int DigitalPinCapture::readInternal(bool *pPinState, uint32_t *pReadings, int count, bool waitForTrigger)
{
    Timer timer;
    
    int lastPinState;
    int lastTransitionTime;
    
    timer.start();    
    if (waitForTrigger)
    {
        while (_signalPin == *pPinState)
        {
            if (timer.read_us() >= _readTimeout) return 0;
        }
        
        while (_signalPin != *pPinState)
        {
            if (timer.read_us() >= _readTimeout) return 0;
        }
        lastTransitionTime = timer.read_us();        
    }
    else
    {
        *pPinState = _signalPin ? true : false;
        lastTransitionTime = timer.read_us();
    }
    
    lastPinState = *pPinState;
    
    int bufferIndex = 0;
    while (bufferIndex < count && timer.read_us() < _readTimeout)
    {
        if (_signalPin != lastPinState)
        {
            uint32_t transitionTime = timer.read_us();
            pReadings[bufferIndex++] = transitionTime - lastTransitionTime;
            lastPinState = !lastPinState;
            lastTransitionTime = transitionTime;            
        }   
    }
    
    timer.stop();        
        
    return bufferIndex;
}

