///////////////////////////////////////////////////////////////////////////////
// Signal Capture Library
// Author: Chris Taylor (taylorza)
#include "mbed.h"
#include "InterruptPinCapture.h"

InterruptPinCapture::InterruptPinCapture(PinName pin, PinMode mode) :
    _started(false),
    _readTimeout(10000000),
    _state(Stopped),
    _signalPin(pin)
{
    _signalPin.mode(mode);        
}

void InterruptPinCapture::setReadTimeout(uint32_t us)
{
    _readTimeout = us;
}

int InterruptPinCapture::read(bool triggerState, uint32_t *pReadings, int count)
{
    return readInternal(&triggerState, pReadings, count, true);
}

int InterruptPinCapture::read(bool *pInitialState, uint32_t *pReadings, int count)
{
    return readInternal(pInitialState, pReadings, count, false);
}

int InterruptPinCapture::readInternal(bool *pPinState, uint32_t *pReadings, int count, bool waitForTrigger)
{
    _bufferIndex = 0;
    _bufferMaxCount = count;
    _pBuffer = pReadings;
    _triggerState = *pPinState ? 1 : 0;   
    
    if (waitForTrigger)
    {
        _state = WaitForTrigger;
    }
    else
    {
        *pPinState = _signalPin.read() ? true : false;
        _state = StartCapturing;
    }
    
    _timer.reset();
    _timer.start();     
    
    _signalPin.rise(this, &InterruptPinCapture::onPinTransition);
    _signalPin.fall(this, &InterruptPinCapture::onPinTransition);
    
    while (_state != Stopped && _timer.read_us() < _readTimeout);
    
    _signalPin.rise(NULL);
    _signalPin.fall(NULL);
    _state = Stopped;        
    _timer.stop();        
        
    return _bufferIndex;
}

void InterruptPinCapture::onPinTransition()
{
    uint32_t transitionTime = (uint32_t)_timer.read_us();
    
    switch(_state)
    {
        case WaitForTrigger:
            if (_signalPin.read() == _triggerState)
            {            
                _state = StartCapturing;            
            }
            break;
            
        case StartCapturing:
            _lastTransitionTime = transitionTime;
            _state = Capturing;            
            break;
            
        case Capturing:
            _pBuffer[_bufferIndex++] = transitionTime - _lastTransitionTime;
            _lastTransitionTime = transitionTime;
            if (_bufferIndex >= _bufferMaxCount)
            {
                _state = Stopped;
            }
            break;        
    }
}
