#include "mbed.h"
#include "GameEngine.h"
#include "GameObject.h"

GameObject::GameObject() : 
    _spriteId(0),
    _position(0,0),    
    _speed(2),
    _animationCounter(0),
    _collisionRect(0, 0, 16, 16)
{
    
}

void GameObject::setPosition(const Point &position)
{
    _position = position;
}

void GameObject::setSpriteId(uint8_t spriteId)
{
    _spriteId = spriteId;
}

void GameObject::setSpeed(uint8_t speed)
{
    _speed = speed;
}

void GameObject::setCollisionRect(uint8_t x, uint8_t y, uint8_t w, uint8_t h)
{
    _collisionRect = Rect(x, y, w, h);
}

void GameObject::animate()
{
    if ((_animationCounter++ % 2) == 0)
    {
        _pParent->animate(_spriteId);
    }
}

bool GameObject::moveLeft()
{    
    if (!canGoLeft()) return false;    
    _position.X -= _speed; 
    return true;
}

bool GameObject::moveRight()
{    
    if (!canGoRight()) return false;
    _position.X += _speed; 
    return true;
}

bool GameObject::moveUp()
{    
    if (!canGoUp()) return false;
    _position.Y -= _speed;    
    return true;
}

bool GameObject::moveDown()
{    
    if (!canGoDown()) return false;    
    _position.Y += _speed;    
    return true;
}

const GameObject* GameObject::detectCollision()
{
    return _pParent->detectCollision(this);
}

bool GameObject::detectCollision(GameObject* other)
{
    return _pParent->detectCollision(this, other);
}

const Block* GameObject::detectBlock()
{
    return _pParent->detectBlock(this);
}

Rect GameObject::getCollisionRect()
{
    return Rect(_position.X + _collisionRect.left, _position.Y + _collisionRect.top, _collisionRect.getWidth(), _collisionRect.getHeight());    
}

bool GameObject::pickupObject()
{    
    uint8_t cellX = _position.X / 8;
    uint8_t cellY = _position.Y / 8;
    for (uint8_t cx = 0; cx < 2; ++cx)
    {
        for (uint8_t cy = 0; cy < 2; ++cy)
        {    
            if (_pParent->pickupObject(cellX + cx, cellY + cy)) return true;
        }
    }
    
    return false;
}

bool GameObject::canGoLeft(uint16_t x, uint16_t y)
{
    int16_t tx = x - _speed;
    if (tx < 0) return false;
        
    if (!_pParent->canEnter(tx, y + 0)) return false;
    if (!_pParent->canEnter(tx, y + 7)) return false;
    if (!_pParent->canEnter(tx, y + 8)) return false;
    if (!_pParent->canEnter(tx, y + 15)) return false;
    
    return true;
}

bool GameObject::canGoRight(uint16_t x, uint16_t y)
{
    int16_t tx = x + 15 + _speed;

    if ((tx / 8) == _pParent->getMapTilesX()) return false;
    
    if (!_pParent->canEnter(tx, y + 0)) return false;
    if (!_pParent->canEnter(tx, y + 7)) return false;
    if (!_pParent->canEnter(tx, y + 8)) return false;
    if (!_pParent->canEnter(tx, y + 15)) return false;
    
    return true;
}

bool GameObject::canGoUp(uint16_t x, uint16_t y)
{
    int16_t ty = y - _speed;
    if (ty < 0) return false;
    
    if (!_pParent->canEnter(x + 4, ty)) return false;
    if (!_pParent->canEnter(x + 7, ty)) return false;
    if (!_pParent->canEnter(x + 8, ty)) return false;
    if (!_pParent->canEnter(x + 11, ty)) return false;
    
    return true;
}

bool GameObject::canGoDown(uint16_t x, uint16_t y)
{   
    int16_t ty = y + 15 + _speed;   
    
    if ((ty / 8) == _pParent->getMapTilesY()) return false;
    
    if (!_pParent->canEnter(x, ty)) return false;
    if (!_pParent->canEnter(x + 7, ty)) return false;
    if (!_pParent->canEnter(x + 8, ty)) return false;
    if (!_pParent->canEnter(x + 15, ty)) return false;
    
    return true;
}

void GameObject::update()
{  
    animate();  
}

void GameObject::draw()
{
    _pParent->drawSprite(_spriteId, _position.X - _pParent->getOffsetX(), _position.Y - _pParent->getOffsetY());    
}
