#include "mbed.h"
#include "BLE.h"
#include "Cadence.h"

#define CONN_INTERVAL 25  /**< connection interval 250ms; in multiples of 0.125ms. (durationInMillis * 1000) / UNIT_0_625_MS; */
#define CONN_SUP_TIMEOUT  8000 /**< Connection supervisory timeout (6 seconds); in multiples of 0.125ms. */
#define SLAVE_LATENCY     0
#define TICKER_INTERVAL   2.0f
#define CADENCE_SUM_NUM 2

#define CADENCE_LOOP_TIME 5

BLE   ble;
Serial pc(USBTX,USBRX);

InterruptIn CadencePin(p30);

Ticker CadenceTicker;
Timer cadence_t;
char cadence_ave = 0;
char cadence_max = 0;
char cadence_sum[CADENCE_SUM_NUM] = {0};
bool cadence_flag = true;
volatile int cadence_num = 0;

void cadence_countUp();
void call_calcCadence();
void CadenceInit();

static const char DEVICENAME[] = "BLE-Nano";
static volatile bool  triggerSensorPolling = false;

const uint8_t MPU6050_service_uuid[] = {
    0x90,0x0f,0xc9,0xd8,0xe5,0x98,0x11,0xe8,0x9f,0x32,0xf2,0x80,0x1f,0x1b,0x9f,0xd1
};

const uint8_t MPU6050_Accel_Characteristic_uuid[] = {
    0x90,0x0f,0xd3,0x7e,0xe5,0x98,0x11,0xe8,0x9f,0x32,0xf2,0x80,0x1f,0x1b,0x9f,0xd1
};

const uint8_t MPU6050_Write_Characteristic_uuid[] =
{
    0x90,0x0f,0xd5,0x04,0xe5,0x98,0x11,0xe8,0x9f,0x32,0xf2,0x80,0x1f,0x1b,0x9f,0xd1
};

uint8_t accelPayload[sizeof(char)*10] = {0,};

uint8_t defaultWriteValue = 10;
uint8_t writePayload[2] = {defaultWriteValue, defaultWriteValue,};


GattCharacteristic  accelChar (MPU6050_Accel_Characteristic_uuid,
                                        accelPayload, (sizeof(char) * 10), (sizeof(char) * 10),
                                        GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

GattCharacteristic  writeChar (MPU6050_Write_Characteristic_uuid,
                                        writePayload, 2, 2,
                                        GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE);

GattCharacteristic *ControllerChars[] = { &accelChar, &writeChar, };
GattService         MPU6050Service(MPU6050_service_uuid, ControllerChars, sizeof(ControllerChars) / sizeof(GattCharacteristic *));



void updateValue(void){
    char   acData[3];
    char   gyData[3];
    char   tempData = 0;
    char   at = 7;
    char   gt = 8;
    char   tickerInterval = 9;
    
    acData[0] = 0;
    acData[1] = 100;
    acData[2] = 2000;
    
    memcpy(accelPayload+sizeof(char)*0, &acData[0], sizeof(acData[0]));
    memcpy(accelPayload+sizeof(char)*1, &acData[1], sizeof(acData[1]));
    memcpy(accelPayload+sizeof(char)*2, &acData[2], sizeof(acData[2]));

    gyData[0] = 3;
    gyData[1] = 4;
    gyData[2] = 5;

    memcpy(accelPayload+sizeof(char)*3, &gyData[0], sizeof(gyData[0]));
    memcpy(accelPayload+sizeof(char)*4, &gyData[1], sizeof(gyData[1]));
    memcpy(accelPayload+sizeof(char)*5, &gyData[2], sizeof(gyData[2]));

    //温度を取得
    //tempData = mpu.getTemp();
    tempData = cadence_ave;
    memcpy(accelPayload+sizeof(char)*6, &tempData, sizeof(tempData));
    
    memcpy(accelPayload+sizeof(char)*7, &at, sizeof(at));
    memcpy(accelPayload+sizeof(char)*8, &gt, sizeof(gt));

    memcpy(accelPayload+sizeof(char)*9, &tickerInterval, sizeof(tickerInterval));

    ble.updateCharacteristicValue(accelChar.getValueAttribute().getHandle(), accelPayload, sizeof(accelPayload));    //Mod
    ble.updateCharacteristicValue(writeChar.getValueAttribute().getHandle(), writePayload, sizeof(writePayload));    //Mod
}

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)    // Mod
{
    
    //DEBUG("Disconnected handle %u, reason %u\n", handle, reason);
    //DEBUG("Restarting the advertising process\n\r");

    ble.startAdvertising();
}

void connectionCallback(const Gap::ConnectionCallbackParams_t *params)
{

    //DEBUG("connected. Got handle %u\r\n", handle);

    /*******************************************************************************/
    /*  CentralがMacOS X の時 connection intervalを設定する場合は                      */
    /*  nRF51822 -> projectconfig.h -> GAP ->                                      */
    /*  CFG_GAP_CONNECTION_MIN_INTERVAL_MS / CFG_GAP_CONNECTION_MAX_INTERVAL_MSを  */
    /*  直接編集すること                                                             */
    /******************************************************************************/
        //Gap::Handle_t handle;
        Gap::ConnectionParams_t gap_conn_params;
        gap_conn_params.minConnectionInterval        = Gap::MSEC_TO_GAP_DURATION_UNITS(CONN_INTERVAL);
        gap_conn_params.maxConnectionInterval        = Gap::MSEC_TO_GAP_DURATION_UNITS(CONN_INTERVAL);
        gap_conn_params.connectionSupervisionTimeout = Gap::MSEC_TO_GAP_DURATION_UNITS(CONN_SUP_TIMEOUT);
        gap_conn_params.slaveLatency                 = SLAVE_LATENCY;
        
    if (ble.updateConnectionParams(params->handle, &gap_conn_params) != BLE_ERROR_NONE) {
        //DEBUG("failed to update connection paramter\r\n");
    }
}

void timeoutCallback(const Gap::TimeoutSource_t source)
{
    //DEBUG("TimeOut\n\r");
    //DEBUG("Restarting the advertising process\n\r");    

    ble.startAdvertising();
}

void periodicCallback(void)
{
    //oneSecondLed = !oneSecondLed; /* Do blinky on LED1 while we're waiting for BLE events */

    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread. */
    triggerSensorPolling = true;
}


void cadence_countUp(){
    if(cadence_flag){
        cadence_flag = false;
        pc.printf("count_up!\n\r");
        cadence_num++;
        wait(0.15);
        cadence_flag = true;
    }
    
}

void call_calcCadence(){
    cadence.calcCadence(cadence_num);
    cadence_num = 0;
}

void CadenceInit(){
    CadencePin.rise(cadence_countUp);
    CadenceTicker.attach(&call_calcCadence, CADENCE_LOOP_TIME);
}

/**************************************************************************/
/*!
    @brief  Program entry point
*/
/**************************************************************************/
int main(void)
{
    CadenceInit();
    
    float ticker_ms = (TICKER_INTERVAL / 100.0f);
    Ticker ticker;
    ticker.attach(periodicCallback, ticker_ms);//0.02f //.2f-sec
    
    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onConnection(connectionCallback);
   // ble.onDataWritten(writtenCallback);
    ble.onTimeout(timeoutCallback);

    /* setup device name */
    ble.setDeviceName((const uint8_t *)DEVICENAME);
    
    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED  | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (const uint8_t *)DEVICENAME, sizeof(DEVICENAME));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                    (const uint8_t *)MPU6050_service_uuid, sizeof(MPU6050_service_uuid));
                                    //(const uint8_t *)MPU6050_adv_service_uuid, sizeof(MPU6050_adv_service_uuid));

    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();

    ble.addService(MPU6050Service);

    while(true) {
        if (triggerSensorPolling && ble.getGapState().connected) {
            triggerSensorPolling = false;
            updateValue();
        } else {
            ble.waitForEvent();
        }
    }
}

