#include <mbed.h>
#include <stdarg.h>
#include "EthernetInterface.h"

#define ECHO_SERVER_PORT  7
#define BUFFER_QUANTITY   (1024*1024)

struct MemoryUsage
{
    uint32_t curFreeRam;
    uint32_t minFreeRam;
    uint32_t maxFreeRam;
    uint32_t avgFreeRam;
    uint32_t stackPointer;
    uint32_t cntSamples;
};

struct MemoryUsage memoryStats = {0,0xFFFFFFFF,0,0,0};

int fakePrintf(void* pvHeapInfo, char const* pFormatString, ...)
{
    struct MemoryUsage * memStat = (struct MemoryUsage *)pvHeapInfo;
    static const char* freeRamFormat = "%d bytes in";
    va_list valist;
    
    if(memcmp(pFormatString,freeRamFormat,strlen(freeRamFormat)) == 0)
    {
        va_start(valist, pFormatString);
        unsigned long freeSize = va_arg(valist, unsigned long);
        memStat->curFreeRam = freeSize;
        if(memStat->minFreeRam > freeSize)
            memStat->minFreeRam = freeSize;
        
        if(memStat->maxFreeRam < freeSize)
            memStat->maxFreeRam = freeSize;
            
        memStat->avgFreeRam = ((memStat->avgFreeRam * memStat->cntSamples) + freeSize)/(++memStat->cntSamples);
    }
    else
    {
        // ignore format
    }
    return 0;
}

void inline printMemoryStats(void)
{
    if(memoryStats.cntSamples == 1)
        printf("\n\n***** Initial Memory Report *****\n");
    else
        printf("\n\n********* Memory Report *********\n");
    
    printf("Current free memory : %d bytes\n",memoryStats.curFreeRam);
    printf("Maximum free memory : %d bytes\n",memoryStats.maxFreeRam);
    printf("Minimum free memory : %d bytes\n",memoryStats.minFreeRam);
    printf("Average free memory : %d bytes\n",memoryStats.avgFreeRam);
    printf("Stack pointer address : %d\n",memoryStats.stackPointer);
    printf("****************************\n");
}

void inline memoryStamp(void)
{
    __heapstats((__heapprt)fakePrintf, &memoryStats);
}

void inline stackPtrSnapshot(void)
{
    memoryStats.stackPointer = __current_sp();
}

int main() {
    
    printf("Ethernet Interface memory test....\n");
    
    // Initial memory status
    memoryStamp();
    stackPtrSnapshot(); // snapshot of the stack pointer before starting benchmark to see the SP.
    printMemoryStats();
    
    EthernetInterface eth;
    int connections = 0;
    eth.init(); //Use DHCP
    eth.connect();
    printf("IP Address %s\n", eth.getIPAddress());
    
    TCPSocketServer server;
    server.bind(ECHO_SERVER_PORT);
    server.listen();
    
    while (true) {
    printf("\nWait for new connection...\n");
    printf("Client number %d\n",++connections);
    TCPSocketConnection client;
    server.accept(client);
    client.set_blocking(false, 1500); // Timeout after (1.5)s
    printf("Connection from: %s\n", client.get_address());
    char buffer[1024];
    while (true) {
        
        int dataReceived = 0;
        int dataSent = 0;
        
        printf("\n\n\nStarting the receiving part...\n");
        while(dataReceived < BUFFER_QUANTITY)
        {
            int n = client.receive(buffer, sizeof(buffer));
            if (n <= 0) {
                printf("Receive error\n");
                break;
            }
            dataReceived += n;
            memoryStamp();
        }
        
       
        printf("Received : %d bytes\nExpected : %d bytes\n",dataReceived,BUFFER_QUANTITY);
        if(dataReceived < BUFFER_QUANTITY)
        {
            printf("Receiving part of the test has failed. Exiting connection.\n");
            break;
        }
        else{
            printf("Receiving has passed...\n");
        }
        
        printf("\n\n\nStarting the sending part...\n");
        while(dataSent < BUFFER_QUANTITY)
        {
            int n = client.send_all(buffer, sizeof(buffer));
            if (n <= 0) {
                printf("Send error\n");
                break;
            }
            dataSent += n;
            memoryStamp();
        }
        
        printf("Sent : %d bytes\nExpected : %d bytes\n",dataSent,BUFFER_QUANTITY);
        if(dataSent < BUFFER_QUANTITY)
        {
            printf("Sending part of the test has failed. Exiting connection.\n");
            break;
        }
        else
        {
            printf("Sending test has passed...\n");
        }
        
        
        printf("\n\n\nStarting echo part...\n");
        dataReceived = dataSent = 0;
        while((dataReceived+dataSent) < 2*BUFFER_QUANTITY)
        {
            int n = client.receive(buffer, sizeof(buffer));
            if (n <= 0) {
                printf("Receive error\n");
                break;
            }
            dataReceived += n;
            
            n = client.send_all(buffer, n);
            if (n <= 0) {
                printf("Send error\n");
                break;
            }
            dataSent += n;
            memoryStamp();
        }
        
        printf("Echo size : %d bytes\nExpected : %d bytes\n",(dataReceived+dataSent),2*BUFFER_QUANTITY);
        if((dataReceived+dataSent) < 2*BUFFER_QUANTITY)
        {
            printf("Echo test has failed.Exiting connection...\n");
        }
        else
        {
            printf("Echo test has passed...\n");
        }
        break;
     }
    client.close();
    printf("Test was finished...\n");
    printMemoryStats();
  }
  
  return 0;
}
