/*
* To enable stack measurement go to pico_mbed.h and add 
*   #define PICO_MEASURE_STACK
* to read the current number of free stack words (integers) call stack_get_free_words
*
* To enable memory measurement go to pico_mbed.h and add
*   #define MEMORY_MEASURE
*/

#include <mbed.h>
#include <stdarg.h>
#include "EthernetInterface.h"

extern "C"
{
    #include "pico_mbed.h"
#ifdef MEMORY_MEASURE
    uint32_t max_mem;
    uint32_t cur_mem;
#endif
}

#define ECHO_SERVER_PORT  7
#define BUFFER_QUANTITY   (1024*1024)

//#define printf(...) do{}while(0)

void printMemoryStats(void)
{
#ifdef PICO_MEASURE_STACK
    printf("******** Stack Statistics\n");
    printf("Current free stack : %d bytes\n",stack_get_free_words()*sizeof(int));
    printf("Total stack size : %d bytes\n",STACK_TOTAL_WORDS*sizeof(int));
    printf("Used : %.2f %%\n",100.0*(float)(STACK_TOTAL_WORDS-stack_get_free_words())/STACK_TOTAL_WORDS);
#endif

#ifdef MEMORY_MEASURE
    printf("******** Memory Statistics\n");
    printf("Current used memory : %d bytes\n",cur_mem);
    printf("Maximum used memory : %d bytes\n",max_mem);
#endif
}

int main() {
    
    printf("Ethernet Interface memory test....\n");
    
    EthernetInterface eth;
    int connections = 0;
    eth.init();
    printf("Waiting for DHCP to give IP\n");
    while(eth.connect() != 0);
    printf("IP Address %s\n", eth.getIPAddress());
    
    TCPSocketServer server;
    server.bind(ECHO_SERVER_PORT);
    server.listen();
    
    while (true) {
    printf("\nWait for new connection...\n");
    printf("Client number %d\n",++connections);
    TCPSocketConnection client;
    server.accept(client);
    client.set_blocking(false, 1500); // Timeout after (1.5)s
    printf("Connection from: %s\n", client.get_address());
    char buffer[1024];
    while (true) {
        
        int dataReceived = 0;
        int dataSent = 0;
        
        while(dataReceived < BUFFER_QUANTITY)
        {
            int n = client.receive(buffer, sizeof(buffer));
            if (n <= 0) {
                printf("Receive error\n");
                break;
            }
            dataReceived += n;
        }
        
        if(dataReceived < BUFFER_QUANTITY)
        {
            printf("Receiving part of the test has failed. Exiting connection.\n");
            printf("Received : %d bytes\nExpected : %d bytes\n",dataReceived,BUFFER_QUANTITY);
            break;
        }
        
        while(dataSent < BUFFER_QUANTITY)
        {
            int n = client.send_all(buffer, sizeof(buffer));
            if (n <= 0) {
                printf("Send error\n");
                break;
            }
            dataSent += n;
        }
        
        if(dataSent < BUFFER_QUANTITY)
        {
            printf("Sending part of the test has failed. Exiting connection.\n");
            printf("Sent : %d bytes\nExpected : %d bytes\n",dataSent,BUFFER_QUANTITY);

            break;
        }
        
        dataReceived = dataSent = 0;
        while((dataReceived+dataSent) < 2*BUFFER_QUANTITY)
        {
            int n = client.receive(buffer, sizeof(buffer));
            if (n <= 0) {
                printf("Receive error\n");
                break;
            }
            dataReceived += n;
            
            n = client.send_all(buffer, n);
            if (n <= 0) {
                printf("Send error\n");
                break;
            }
            dataSent += n;
        }
        
        if((dataReceived+dataSent) < 2*BUFFER_QUANTITY)
        {
            printf("Echo size : %d bytes\nExpected : %d bytes\n",(dataReceived+dataSent),2*BUFFER_QUANTITY);
            printf("Echo test has failed.Exiting connection...\n");
        }
 
        break;
     }
 
    client.close();
    printf("Test was finished...\n");
    printMemoryStats();
  }
  
  return 0;
}
