// LoadCell_STM32_SDRAM main
// (C) Tapio Valli 2018-02-24

#include "mbed.h"
#include "mbed_stats.h"

#include "LCD_DISCO_F746NG.h"
#include "SDRAM_DISCO_F746NG_BlockDevice.h"
#include "FATFileSystem.h"

// Physical block device, can be any device that supports the BlockDevice API
#define BLOCK_SIZE 512

// SDRAM Block Device
SDRAM_DISCO_F746NG_BlockDevice bd;

// File system declaration
FATFileSystem fs("fs");

// define the Serial object
Serial pc(USBTX, USBRX);
LCD_DISCO_F746NG lcd;

typedef struct {
  uint16_t *array;
  size_t used;
  size_t size;
} Array;

// Prototypes

extern void initArray(Array *a, size_t initialSize);
extern void insertArray(Array *a, uint32_t newsize);
extern void freeArray(Array *a);

extern size_t string_parser(char *input, char ***word_array);
extern void PlotData(uint32_t XCoordinate,uint32_t YCoordinate);
extern void InitScreen(uint32_t BackGroundColor,uint32_t ForeGroundColor);
extern void LCDWrite(uint32_t Line,char Str[],Text_AlignModeTypdef AlingMode);
extern void CountDown(uint32_t millisecs);
extern void SamplePoints(Array *Data,uint32_t NoOfPoints,uint32_t Period_us);
extern void AvgAndPlotPoints(Array *Data,uint32_t NoOfPoints, uint32_t AvgSize);

extern void BlockDeviceErase();
extern void MountFS();
extern void WriteData2FS(Array *Data,uint32_t NoOfPoints,uint32_t FileNumber);
extern void ReadFile2Data(Array *Data,uint32_t NoOfPoints,uint32_t FileNumber);
extern void DirListing(void);

int main() {
    
    FMC_SDRAM_CommandTypeDef SDRAMCommandStructure;
    
    // Initialize SDRAM
    bd.init();
    
    // Issue self-refresh command to SDRAM device
    SDRAMCommandStructure.CommandMode            = FMC_SDRAM_CMD_SELFREFRESH_MODE;
    SDRAMCommandStructure.CommandTarget          = FMC_SDRAM_CMD_TARGET_BANK2;
    SDRAMCommandStructure.AutoRefreshNumber      = 1;
    SDRAMCommandStructure.ModeRegisterDefinition = 0;
    if (bd.Sendcmd(&SDRAMCommandStructure) != HAL_OK) 
    {
      error("BSP_SDRAM_Sendcmd FAILED\n");
    }
    
    // Erase in any case for start
    BlockDeviceErase();

    // Try to mount the filesystem
    MountFS();
    
    char CmdBuffer[30];
    char Arg[30]=" ";
    char Cmd[30]=" ";
    size_t n=0;

    uint32_t    NoOfPoints=19200;  
    uint32_t    AvgSize=10;
    uint32_t    Period_us=100;
    uint32_t    Count_ms=1000;
    uint32_t    MeasNumber=1;
    int err;
    
    mbed_stats_heap_t heap_stats;
        
    // Clear screen, set it up 
    InitScreen(LCD_COLOR_BLACK,LCD_COLOR_WHITE);
        
    //Initialize data storage
    Array Data;
    initArray(&Data,NoOfPoints);  // initially 19200 elements
    
    InitScreen(LCD_COLOR_BLACK,LCD_COLOR_WHITE);
    LCDWrite(5,"Ready.",CENTER_MODE);
    
    while (!((strcmp(Cmd,"quit")==0)&&(n==1))) {
        
        // Print Ready and current settings        
        pc.printf(
        "Ready. Settings are Points=%u, Avg=%u, Period_us=%u, "
        "Count_ms=%u. Sampling will take apprx. %.3f secs\r\n",
        NoOfPoints,AvgSize,Period_us,Count_ms,
        (float)(NoOfPoints*Period_us/1000000.0));
                   
        mbed_stats_heap_get(&heap_stats);
        pc.printf("Current heap = %lu Max heap = %lu\r\n", 
        heap_stats.current_size,heap_stats.max_size);
                                                                           
        pc.gets(CmdBuffer,30);
        pc.printf("I got %s \r\n", CmdBuffer);
        strcpy(Cmd," ");
        strcpy(Arg," ");
                
        // Parse command and possible numeric arg
        char s[] = "Initial string";
        char ** word_array=NULL;
        
        strcpy(s,CmdBuffer);
        n=string_parser(s,&word_array);

        for ( size_t i=0;i<n;i++) {
            if (i==0) {strcpy(Cmd,word_array[i]);}
            if (i==1) {strcpy(Arg,word_array[i]);}
            if (i>1) {pc.printf("Wrong number of arguments \r\n");}
        }
        
        pc.printf("Cmd = %s Arg = %s \r\n",Cmd,Arg);
        for ( size_t i=0;i<n;i++) free( word_array[i] );
        free(word_array);
    
        // Branch based on command        
        // meas: Sample and plot a data set
        if((strcmp(Cmd,"meas")==0)&&(n==1)) {
                        
            // Countdown
            CountDown(Count_ms);
                                 
            // Sample & plot data one time
            SamplePoints(&Data,NoOfPoints,Period_us);
            AvgAndPlotPoints(&Data,NoOfPoints,AvgSize);
            WriteData2FS(&Data,NoOfPoints,MeasNumber);
            MeasNumber++;
            
            //-- implement http server with fs as storage
            //-- connect and pull data from there
            //-- implement disk init as functions & commands
        }
        
        // setpoints: Adjust sampled points                          
        else if((strcmp(Cmd,"setpoints")==0)&&(n==2)) {
            // Allocate more or less data space
            NoOfPoints=(uint32_t) strtol(Arg,NULL,10);
            pc.printf("Old Data size is %u New NoOfPOints = %u \r\n",Data.size,NoOfPoints);
            insertArray(&Data,NoOfPoints);
            pc.printf("New Array size is %u \r\n",Data.size);
        }
            
        // setavg: Adjust average amount in samples            
        else if((strcmp(Cmd,"setavg")==0)&&(n==2)) {
            AvgSize=(uint32_t) strtol(Arg,NULL,10);
        }
            
        // setperiod: Adjust sample period in us
        else if((strcmp(Cmd,"setperiod")==0)&&(n==2)) {
            Period_us=(uint32_t) strtol(Arg,NULL,10);
        }
        
        // setcount: Adjust countdown period in ms
        else if((strcmp(Cmd,"setcount")==0)&&(n==2)) {
            Count_ms=(uint32_t) strtol(Arg,NULL,10);
        }
        
        // dir: List files
        else if((strcmp(Cmd,"dir")==0)&&(n==1)) {
            DirListing();
        }
        
        // quit: Exit on next while
        else if((strcmp(Cmd,"quit")==0)&&(n==1)) {
            // Do nothing yet
        }
        else {
            pc.printf("Wrong command or argument \r\n");
        }
    }
    
    // Free memory after quit
    freeArray(&Data);
    
    // Tidy up filesystem
    pc.printf("Unmounting filesystem.");
    fflush(stdout);
    err = fs.unmount();
    pc.printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
    if (err < 0) {
        error("error: %s (%d)\n", strerror(-err), err);
    }

    // Print informative messages
    InitScreen(LCD_COLOR_BLACK,LCD_COLOR_WHITE);
    LCDWrite(5,"Stop.",CENTER_MODE);    
    pc.printf("Exit. Data freed. Stop. \r\n");
    
    // Infinite loop
    while(1) 
    {
        // Do nothing
    }
}
