// LoadCell_STM32_RAM v2 functions
// (C) Tapio Valli 2018-02-17

#include "main.h"

int err;

void initArray(Array *a, size_t initialSize) {
  a->array = (uint16_t *)malloc(initialSize * sizeof(uint16_t));
  a->used = 0;
  a->size = initialSize;
}

void insertArray(Array *a, uint32_t newsize) {
    a->size = newsize;
    a->array = (uint16_t *)realloc(a->array, a->size * sizeof(uint16_t));
    a->used = 0;
}

void freeArray(Array *a) {
  free(a->array);
  a->array = NULL;
  a->used = a->size = 0;
}

size_t string_parser(char *input, char ***word_array) 
{
    size_t n = 0;
    const char *p = input;

    while ( *p )
    {
        while ( isspace( ( unsigned char )*p ) ) ++p;
        n += *p != '\0';
        while ( *p && !isspace( ( unsigned char )*p ) ) ++p;
    }

    if ( n )
    {
        size_t i = 0;

        *word_array = (char**)malloc( n * sizeof( char * ) ); 

        p = input;

        while ( *p )
        {
            while ( isspace( ( unsigned char )*p ) ) ++p;
            if ( *p )
            {
                const char *q = p;
                while ( *p && !isspace( ( unsigned char )*p ) ) ++p;

                size_t length = p - q;

                ( *word_array )[i] = ( char * )malloc( length + 1 );

                strncpy( ( *word_array )[i], q, length );
                ( *word_array )[i][length] = '\0';

                ++i;
            }
        }           
    }

    return n;
}  

void PlotData(uint32_t XCoordinate,uint32_t YCoordinate)
{
    // Plot at x,y
    lcd.DrawHLine(XCoordinate,YCoordinate,1);        
}

void InitScreen(uint32_t BackGroundColor,uint32_t ForeGroundColor)
{

// #define LCD_COLOR_BLUE          ((uint32_t)0xFF0000FF)
// #define LCD_COLOR_GREEN         ((uint32_t)0xFF00FF00)
// #define LCD_COLOR_RED           ((uint32_t)0xFFFF0000)
// #define LCD_COLOR_CYAN          ((uint32_t)0xFF00FFFF)
// #define LCD_COLOR_MAGENTA       ((uint32_t)0xFFFF00FF)
// #define LCD_COLOR_YELLOW        ((uint32_t)0xFFFFFF00)
// #define LCD_COLOR_LIGHTBLUE     ((uint32_t)0xFF8080FF)
// #define LCD_COLOR_LIGHTGREEN    ((uint32_t)0xFF80FF80)
// #define LCD_COLOR_LIGHTRED      ((uint32_t)0xFFFF8080)
// #define LCD_COLOR_LIGHTCYAN     ((uint32_t)0xFF80FFFF)
// #define LCD_COLOR_LIGHTMAGENTA  ((uint32_t)0xFFFF80FF)
// #define LCD_COLOR_LIGHTYELLOW   ((uint32_t)0xFFFFFF80)
// #define LCD_COLOR_DARKBLUE      ((uint32_t)0xFF000080)
// #define LCD_COLOR_DARKGREEN     ((uint32_t)0xFF008000)
// #define LCD_COLOR_DARKRED       ((uint32_t)0xFF800000)
// #define LCD_COLOR_DARKCYAN      ((uint32_t)0xFF008080)
// #define LCD_COLOR_DARKMAGENTA   ((uint32_t)0xFF800080)
// #define LCD_COLOR_DARKYELLOW    ((uint32_t)0xFF808000)
// #define LCD_COLOR_WHITE         ((uint32_t)0xFFFFFFFF)
// #define LCD_COLOR_LIGHTGRAY     ((uint32_t)0xFFD3D3D3)
// #define LCD_COLOR_GRAY          ((uint32_t)0xFF808080)
// #define LCD_COLOR_DARKGRAY      ((uint32_t)0xFF404040)
// #define LCD_COLOR_BLACK         ((uint32_t)0xFF000000)
// #define LCD_COLOR_BROWN         ((uint32_t)0xFFA52A2A)
// #define LCD_COLOR_ORANGE        ((uint32_t)0xFFFFA500)
// #define LCD_COLOR_TRANSPARENT   ((uint32_t)0xFF000000)

    lcd.Clear(BackGroundColor);
    lcd.SetBackColor(BackGroundColor);
    lcd.SetTextColor(ForeGroundColor);
    lcd.SetFont(&Font20);
}

void LCDWrite(uint32_t Line,char Str[],Text_AlignModeTypdef AlignMode)
{
    char IntStr[50];
    
//    InitScreen(LCD_COLOR_BLACK,LCD_COLOR_WHITE,Font20);
    
    lcd.ClearStringLine(Line);   
    snprintf(IntStr,50,Str);
    lcd.DisplayStringAt(0, LINE(Line), (uint8_t *)IntStr, AlignMode);
}

void CountDown(uint32_t millisecs)
{
    InitScreen(LCD_COLOR_BLACK,LCD_COLOR_WHITE);

    LCDWrite(5," ",CENTER_MODE);
    wait_ms(1);

    LCDWrite(5,"Starting in 3... ",CENTER_MODE);
    wait_ms(millisecs);
    
    LCDWrite(5,"Starting in 2... ",CENTER_MODE);
    wait_ms(millisecs);

    LCDWrite(5,"Starting in 1... ",CENTER_MODE);
    wait_ms(millisecs);
    
    InitScreen(LCD_COLOR_BLACK,LCD_COLOR_WHITE);
    LCDWrite(5,"GO!",CENTER_MODE);
}

void SamplePoints(Array *Data,uint32_t NoOfPoints,uint32_t Period_us)
{
    AnalogIn    ain(A0);
    uint32_t i;
  
    // Measure NoOfPoints values (f.ex. 19200) 
    for(i=0;i<NoOfPoints;i++) {
        Data->array[i]=(uint16_t)(ain.read_u16());
        wait_us(Period_us);
    }

    InitScreen(LCD_COLOR_BLACK,LCD_COLOR_WHITE);
    LCDWrite(5,"DONE!",CENTER_MODE);

    pc.printf("Sampling done.\r\n",i);
}

void AvgAndPlotPoints(Array *Data,uint32_t NoOfPoints, uint32_t AvgSize) {
    
    uint32_t i1,i2;
    
    uint32_t BufferSum,BufferAvg;
    uint32_t XCoord,YCoord;
    char MyStr[50];
    
    pc.printf("Averaging... \r\n");
    InitScreen(LCD_COLOR_BLACK,LCD_COLOR_RED);  

    for(i1=0;i1<NoOfPoints;i1++) {
        BufferSum=0;
        
        // Read AvgSize samples
        for(i2=i1;i2<i1+AvgSize;i2++) {
            BufferSum=BufferSum+(uint32_t)Data->array[i2];
        }
        
        BufferAvg=BufferSum/AvgSize;

        // Calculate two coords and plot
        XCoord=((i1*480.0)/NoOfPoints);
        YCoord=(272.0*(BufferAvg/65536.0));
        
        PlotData(XCoord,YCoord);
    }  
    
    pc.printf("Averaging done, Points = %u Avg = %u \r\n", i1,AvgSize);

    LCDWrite(0,"",CENTER_MODE);
    snprintf(MyStr,50,"Pnts = %d Avg = %d",NoOfPoints,AvgSize);
    LCDWrite(0,MyStr,RIGHT_MODE);
}

void BlockDeviceErase() {
    pc.printf("Initializing the block device... ");
    fflush(stdout);
    int err = bd.init();
    pc.printf("%s\n", (err ? "Fail" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }
    
    pc.printf("Erasing the block device... ");
    fflush(stdout);
    err = bd.erase(0, bd.size());
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }
    pc.printf("Device size is %d\n",bd.size());
    
    printf("Deinitializing the block device... ");
    fflush(stdout);
    err = bd.deinit();
    printf("%s\n", (err ? "Fail" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }
}

void MountFS() {
    pc.printf("Mounting the filesystem... ");
    fflush(stdout);
    int err = fs.mount(&bd);
    pc.printf("%s\n", (err ? "Fail" : "OK"));
    if (err) {
        // Reformat if we can't mount the filesystem
        // this should only happen on the first boot
        pc.printf("No filesystem found, formatting... ");
        fflush(stdout);
        err = fs.reformat(&bd,bd.get_program_size());
        pc.printf("%s\n", (err ? "Fail" : "OK"));
        if (err) {
            error("error: %s (%d)\n", strerror(-err), err);
        }
    }
}

void WriteData2FS(Array *Data,uint32_t NoOfPoints,uint32_t FileNumber) {
    
    char    FName[30];
    uint32_t idx;
    
    // Build the filename
    snprintf(FName,30,"/fs/meas%d.txt",FileNumber);
        
    // Open the data file
    pc.printf("Opening %s ... ",FName);
    fflush(stdout);
    FILE *f = fopen(FName, "r+");

    pc.printf("%s\n", (!f ? "Fail" : "OK"));
    if (!f) {
        // Create the data file if it doesn't exist
        pc.printf("No file found, creating a new file... ");
        fflush(stdout);
        f = fopen(FName, "w+");
        pc.printf("%s\n", (!f ? "Fail" : "OK"));
        if (!f) {
            error("error: %s (%d)\n", strerror(errno), -errno);
            return;
        }
    }

    // Write data points to file
    pc.printf("Writing data ... ");
    for (idx=0;idx<NoOfPoints;idx++) {
        err=fprintf(f,"%d\n",(uint16_t)Data->array[idx]);
        if (err < 0) {
            pc.printf("Fail\n");
            error("error: %s (%d)\n", strerror(errno), -errno);
            return;
            
        }
    }
    pc.printf("\rWriting data done.");
    pc.printf("\r File size is %d",GetFileSize(f));
    
    // Close the file which also flushes any cached writes
    pc.printf(" Closing %s ... ",FName);
    fflush(stdout);
    err = fclose(f);
    printf("%s\n", (err < 0 ? "Fail" : "OK"));
    if (err < 0) {
        error("error: %s (%d)\n", strerror(errno), -errno);
        return;
    }
}

void ReadFile2Data(Array *Data,uint32_t NoOfPoints,uint32_t FileNumber) {

    char     FName[30];
    uint32_t idx;
    
    // Build the filename
    snprintf(FName,30,"/fs/meas%d.txt",FileNumber);
        
    // Open the data file for reading
    pc.printf("Opening %s ... ",FName);
    fflush(stdout);
    FILE *f = fopen(FName, "r");

    pc.printf("%s\n", (!f ? "Fail" : "OK"));
    if (!f) {
        // Print error and return
        pc.printf("File not found. Check name.");
        return;
    }
    
    pc.printf("Seeking file... ");
    fflush(stdout);
    err = fseek(f, 0, SEEK_SET);
    printf("%s\n", (err < 0 ? "Fail" : "OK"));
    if (err < 0) {
        error("error: %s (%d)\n", strerror(errno), -errno);
        return;
    }

    // Go through and increment the numbers
    pc.printf("Reading data... ");
    for(idx=0;idx<NoOfPoints;idx++) {
        // Get current stream position
        long pos=ftell(f);

        // Parse out the number and put it in data array
        uint32_t number;
        fscanf(f,"%d",&number);
        Data->array[idx]=number;
    }

    pc.printf("\rReading data done. ");
    
    // Close the file which also flushes any cached writes
    pc.printf("Closing %s ... ",FName);
    fflush(stdout);
    err = fclose(f);
    pc.printf("%s\n", (err < 0 ? "Fail" : "OK"));
    if (err < 0) {
        error("error: %s (%d)\n", strerror(errno), -errno);
        return;
    }
}

void DirListing() {

    // Display the root directory
    pc.printf("Opening the root directory... ");
    fflush(stdout);
    DIR *d = opendir("/fs/");
    pc.printf("%s\n", (!d ? "Fail" : "OK"));
    if (!d) {
        error("error: %s (%d)\n", strerror(errno), -errno);
        return;
    }

    pc.printf("Root directory:\n");
    while (true) {
        struct dirent *e = readdir(d);
        if (!e) {
            break;
        }

        pc.printf("    %s\n", e->d_name);
    }

    pc.printf("Closing the root directory... ");
    fflush(stdout);
    err = closedir(d);
    pc.printf("%s\n", (err < 0 ? "Fail" : "OK"));
    if (err < 0) {
        error("error: %s (%d)\n", strerror(errno), -errno);
        return;
    }
}

uint32_t GetFileSize(FILE *fp) {
    fseek(fp, 0, SEEK_END);
    int size = ftell(fp);
    fseek(fp, 0, SEEK_SET);
    
    return size;
}
