/* An I2C text LCD library for Displaytronic ACM1602NI-FLW-FBW-M01
 * Copyright 2013, Takuo WATANABE (wtakuo)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef ACM1602NI_H
#define ACM1602NI_H
 
#include "mbed.h"
 
/** An I2C text LCD library for Displaytronic ACM1602NI-FLW-FBW-M01.
 * The device does not work with default I2C library due to its slow I2C responce.
 * This library adds some extra waits so that the device can answer to the I2C commands.
 * The interface is basically the same as TextLCD by Simon Ford.
 *
 * @code
 * #include "mbed.h"
 * #include "ACM1602NI.h"
 *
 * // p9: sda, p10: scl
 * ACM1602NI lcd(p9, p10);
 *
 * int main() {
 *     lcd.printf("Hello, World!\n");
 *     lcd.printf("ACM1602NI\n");
 * }
 * @endcode
 */
class ACM1602NI : public Stream
{
public:
    /** Create an ACM1602NI object connected to the specified I2C pins.
     *
     * @param sda   The I2C data pin
     * @param scl   The I2C clock pin
     */
    ACM1602NI(PinName sda, PinName scl);
 
    /** Create an ACM1602NI object connected to the specified I2C port.
     *
     * @param i2c   The I2C port to connect to
     */
    ACM1602NI(I2C &i2c);
 
    /** Initialize the device and clear screen
     */
    void init();
 
    /** Locate to a screen column and row
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @param row     The vertical position from the top, indexed from 0
     */
    void locate(int column, int row);
 
    /** Clear the screen and locate to 0,0 */
    void cls();
 
    int rows();
    int columns();
 
protected:
    virtual int _putc(int value);
    virtual int _getc();
 
    int address(int column, int raw);
    void character(int column, int row, int c);
    int writeBytes(const char *data, int length, bool repeated = false);
    void writeCommand(int command);
    void writeData(int data);
 
    static const int i2c_addr = 0x50 << 1;
    static const int i2c_bit_wait_us = 20;
    static const int i2c_command_wait_ms = 4;
 
    static const int display_columns = 16;
    static const int display_rows = 2;
 
    I2C _i2c;
    int _column, _row;
};
 
#endif
 
