/* A library for STMicroelectronics STTS751 I2C temperature sensor
 * Copyright 2014, Takuo WATANABE (wtakuo)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef STTS751_H
#define STTS751_H

#include "mbed.h"

/** A library for STMicroelectronics STTS751 I2C temperature sensor
 *
 * http://www.st.com/web/catalog/sense_power/FM89/SC294/PF220116
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "STTS751.h"
 *
 * // I2C pins: p9 = sda, p10 = scl
 * STTS751 sensor(p9, p10);
 *
 * // You can specify an I2C object instead.
 * // I2C i2c(p2, p10);
 * // STTS751 sensor(i2c);
 * 
 * int main() {
 *     // set the temperature resolution to 12bits
 *     sensor.setResolution(STTS751::RES_12);
 *     while (true) {
 *         printf("tmp: %.4f\n", (float)sensor);
 *         wait(1);
 *     }
 * }
 * @endcode
 */
class STTS751 {
public:
    /** Device models
     */
    enum Model {
        MODEL_0 = 0x00, /**< Model 0 (default) */
        MODEL_1 = 0x40  /**< Model 1 */
    };

    /** I2C slave address selectors
     * The I2C slave address of the device is determined by the pull up register for Addr/~Therm pin.
     *
     */
    enum Address {
        ADDRESS_0 = (0x48 << 1), /**< pull up resistor = 7.5K */
        ADDRESS_1 = (0x49 << 1), /**< pull up resistor = 12K */
        ADDRESS_2 = (0x38 << 1), /**< pull up resistor = 20K */
        ADDRESS_3 = (0x39 << 1)  /**< pull up resistor = 33K or GND (default) */
    };

    /** Temperature resolutions
     */
    enum Resolution {
        RES_9 = 0x08, /**<  9 bits */
        RES_10 = 0x00, /**< 10 bits (default) */
        RES_11 = 0x04, /**< 11 bits */
        RES_12 = 0x0c /**< 12 bits */
    };

    /** Create an STTS751 object connected to the specified I2C pins.
     *
     * @param sda     I2C data pin
     * @param scl     I2C clock pin
     * @param standby Standby mode (defaults to false)
     * @param addr    I2C slave address (defaults to ADDRESS_3)
     * @param model   Device model (defaults to MODEL_0)
     */
    STTS751(PinName sda, PinName scl,
            bool standby = false, Address addr = ADDRESS_3, Model model = MODEL_0);


    /** Create an STTS751 object connected to the specified I2C port.
     *
     * @param i2c     I2C port
     * @param standby Standby mode (defaults to false)
     * @param addr    I2C slave address (defaults to ADDRESS_3)
     * @param model   Device model (defaults to MODEL_0)
     */
    STTS751(I2C &_i2c,
            bool standby = false, Address addr = ADDRESS_3, Model model = MODEL_0);

    /** Initialize the device
     */
    void init();

    /** The I2C slave address (8bit) of the device
     *
     * @returns  The I2C slave address (8bit) of the device
     */
    int addr();

    /** Get the current temperature resolution
     *
     * @returns  The current temperature resolution
     */
    Resolution resolution();

    /** Set the temperature resolution
     *
     * @param res  Resolution (defaults to RES_10)
     */
    void setResolution(Resolution res = RES_10);

    /** Get the current conversion rate
     *
     * @returns  The current conversion rate
     */
    int conversionRate();

    /** Set the conversion rate
     *  Set the number of times the temperature value is updated each second.
     *  The conversion rate (times/second) can be calculated by 2^(rate - 4).
     *  Note that rate should be less than 8 if the temperature resolution is 12-bit and
     *  should be less than 9 if the temperature resolution is greater than 10-bit
     *
     * @param rate  Conversion rate (0 .. 9)
     */
    void setConversionRate(int rate);

    /** Set the device mode
     *
     * @param standby  true : standby mode, false : continuous conversion mode
     */
    void setStandbyMode(bool standby);

    /** Start a temperature conversion (in standby mode)
     */
    void start();

    /** Checks that a temperature conversion has finished and the temperature value is available
     *
     * @returns  true if a temperature conversion has finished and the temperature value is available
     */
    bool ready();

    /** Obtain the current temperature measurement
     *
     * @param nowait  If true, read the temperature value without waiting for finishing a conversion (in standby mode)
     * @returns       The current temperature measurement in Celsius.
     */
    float temp(bool nowait = false);

#ifdef MBED_OPERATORS
    /** A shorthand for temp()
     *
     * @returns  The current temperature measurement in Celsius.
     */
    operator float();
#endif

protected:
    /** I2C registers
     */
    enum {
        REG_TEMPERATURE_H = 0x00,
        REG_STATUS = 0x01,
        REG_TEMPERATURE_L = 0x02,
        REG_CONFIGURATION = 0x03,
        REG_CONV_RATE = 0x04,
        REG_HIGH_LIMIT_H = 0x05,
        REG_HIGH_LIMIT_L = 0x06,
        REG_LOW_LIMIT_H = 0x07,
        REG_LOW_LIMIT_L = 0x08,
        REG_ONESHOT = 0x0f,
        REG_THERM = 0x20,
        REG_THERM_HYSTERESIS = 0x21,
        REG_SMBUS_TIMEOUT = 0x22,
        REG_PRODUCT_ID = 0xfd,
        REG_MFG_ID = 0xfe,
        REG_REVISION_ID = 0xff,
    };

    /** Configuration register (default 0x00)
     * - b7 : MASK1
     * - b6 : ~RUN/STOP (0 : continuous conversion mode, 1 : standby mode)
     * - b5 : 0
     * - b4 : RFU
     * - b3 : Tres1
     * - b2 : Tres0
     * - b1-b0 : RFU
     */
    enum {
        CONF_MASK1 = 0x80,
        CONF_RUNSTOP = 0x40,
        CONF_RES_MASK = 0x0c,
    };

    /** Status Register (default undefined)
     * - b7 : Busy
     * - b6 : T_HIGH
     * - b5 : T_LOW
     * - b4-b1 : RFU
     * - b0 : THRM
     */
    enum {
        STATUS_BUSY = 0x80,
        STATUS_THIGH = 0x40,
        STATUS_LOW = 0x20,
        STATUS_THURM = 0x01
    };

    /** Conversion Rate Register (default 4)
     * - b7-b4: 0
     * - b3-b0: Conversion rate (0..9)
     */
    enum {
        CONV_RATE_MASK = 0x0f
    };

    I2C _i2c;
    const int _addr;
    bool _standby;

    char read8(char reg);
    void write8(char reg, char data);
};

#endif
