/* A library for STMicroelectronics STTS751 I2C temperature sensor
 * Copyright 2014, Takuo WATANABE (wtakuo)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "STTS751.h"

#define SET_BITS(x,m,b) (((x) & ~(m)) | (b))

STTS751::STTS751(PinName sda, PinName scl, bool standby, Address addr, Model model) :
        _i2c(sda, scl), _addr(addr | model), _standby(standby) {
    init();
}

STTS751::STTS751(I2C &i2c, bool standby, Address addr, Model model) :
        _i2c(i2c), _addr(addr | model), _standby(standby) {
    init();
}

void STTS751::init() {
    char conf = 0x00;
    if (_standby)
        conf |= CONF_RUNSTOP;
    write8(REG_CONFIGURATION, conf);
    // conversion rate = 1/sec
    write8(REG_CONV_RATE, 0x04);
}

int STTS751::addr() {
    return _addr;
}

STTS751::Resolution STTS751::resolution() {
    char conf = read8(REG_CONFIGURATION);
    return (STTS751::Resolution) (conf & CONF_RES_MASK);
}

void STTS751::setResolution(STTS751::Resolution res) {
    char conf = read8(REG_CONFIGURATION);
    conf = SET_BITS(conf, CONF_RES_MASK, res);
    write8(REG_CONFIGURATION, conf);
}

int STTS751::conversionRate() {
    char conv = read8(REG_CONV_RATE);
    return conv & CONV_RATE_MASK;
}

void STTS751::setConversionRate(int rate) {
    write8(REG_CONV_RATE, (char)(rate & CONV_RATE_MASK));
}

void STTS751::setStandbyMode(bool standby) {
    _standby = standby;
    char conf = read8(REG_CONFIGURATION);
    if (_standby)
        conf |= CONF_RUNSTOP;
    else
        conf &= ~CONF_RUNSTOP;
    write8(REG_CONFIGURATION, conf);
}

void STTS751::start() {
    if (ready())
        write8(REG_ONESHOT, 1);
}

bool STTS751::ready() {
    char status = read8(REG_STATUS);
    return (status & STATUS_BUSY) == 0;
}

float STTS751::temp(bool nowait) {
    if (_standby && !nowait) {
        start();
        while (!ready())
            wait(0.01);
    }
    signed char h = read8(REG_TEMPERATURE_H);
    unsigned char l = read8(REG_TEMPERATURE_L);
    return ((h << 8) | l) / 256.0;
}

#ifdef MBED_OPERATORS
STTS751::operator float() {
    return temp();
}
#endif

char STTS751::read8(char reg) {
    _i2c.write(_addr, &reg, 1, true);
    _i2c.read(_addr, &reg, 1);
    return reg;
}

void STTS751::write8(char reg, char data) {
    char buff[2] = { reg, data };
    _i2c.write(_addr, buff, 2);
}
