/* A digital input with switch debouncing
 * Copyright 2015, Takuo Watanabe
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef DEBOUNCEIN_H_
#define DEBOUNCEIN_H_

#include "mbed.h"

/** A digital input with switch debouncing
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "DebounceIn.h"
 *
 * int nrise = 0, nfall = 0;
 * void rise() { nrise++; }
 * void fall() { nfall++; }
 *
 * int main() {
 *     DebounceIn button(p14);
 *     button.rise(rise);
 *     button.fall(fall);
 *     while (true) {
 *         printf("nrise=%d, nfall=%d\n", nrise, nfall);
 *         wait(1);
 *     }
 * }
 * @endcode
 */
class DebounceIn: public mbed::DigitalIn {
public:
    /** Create a DebounceIn object connected to the specified pin
     *
     * @param pin  DebounceIn pin to connect to
     * @param us   Input sampling rate (default 1000 microseconds)
     */
    DebounceIn(PinName pin, timestamp_t us = 1000);

    virtual ~DebounceIn();

    /** Set the input sampling rate
     *
     * @param us  Input sampling rate
     */
    void set_sample_rate(timestamp_t us);

    /** Read the input, represented as 0 or 1 (int)
     *
     * @returns  An integer representing the state of the input pin
     */
    int read(void);

#ifdef MBED_OPERATORS
    /** An operator shorthand for read()
     */
    operator int();
#endif

    /** Attach a function to call when a rising edge occurs on the input
     *
     * @param fptr pointer to a void function, or 0 to set as none
     */
    void rise(void (*fptr)(void));

    /** Attach a member function to call when a rising edge occurs on the input
     *
     *  @param tptr pointer to the object to call the member function on
     *  @param mptr pointer to the member function to be called
     */
    template<typename T>
    void rise(T* tptr, void (T::*mptr)(void)) {
        _rise.attach(tptr, mptr);
    }

    /** Attach a function to call when a falling edge occurs on the input
     *
     *  @param fptr pointer to a void function, or 0 to set as none
     */
    void fall(void (*fptr)(void));

    /** Attach a member function to call when a falling edge occurs on the input
     *
     *  @param tptr pointer to the object to call the member function on
     *  @param mptr pointer to the member function to be called
     */
    template<typename T>
    void fall(T* tptr, void (T::*mptr)(void)) {
        _fall.attach(tptr, mptr);
    }

private:
    Ticker _ticker;
    int _prev, _curr;
    FunctionPointer _rise, _fall;
    void sample(void);
};

#endif /* DEBOUNCEIN_H_ */
