#ifndef DRV2667_H
#define DRV2667_H

#include "mbed.h"

// DRV2667 Slave Address
#define SLAVE_ADDR_7_BIT 0x59

/** Library for the TI DRV2667
 *
 */

class DRV2667
{

public:
    // Selects the source to be played in register 0x01, bit 2
    enum InputMux {
        Digital = 0,    // Digital input source
        Analog = 0x04,  // Analog input source
    };

    // Selects the gain for the amplifier in register 0x01, bits [1:0]
    enum Gain {
        GNx1,    // 0x00:  50 Vpp - 28.8 dB
        GNx2,    // 0x01: 100 Vpp - 34.8 dB
        GNx3,    // 0x02: 150 Vpp - 38.4 dB
        GNx4,    // 0x03: 200 Vpp - 40.7 dB
    };

    // Time period when the FIFO runs empty and the device goes into idle
    // mode, powering down the boost converter and amplifier in register
    // 0x02, bits [3:2]
    enum Timeout {
        TOx1 = 0,    // 5 msec
        TOx2 = 0x04,    // 10 msec
        TOx3 = 0x08,    // 15 msec
        TOx4 = 0x0C,    // 20 msec
    };

    /**
    * Create a DRV2667 object
    *
    * @param &i2c pointer of I2C object
    * @param im DRV2667::Digital or DRV2667::Analog
    * @param gn gain for the amplifier
    * @param to timeout for FIFO interface (digital input only)
    */
    DRV2667(I2C &i2c, InputMux im = Digital, Gain gn = GNx1, Timeout to = TOx1);

    /**
    * Write value to specified register of device
    * @param reg      The device register to write
    * @param value    The value to write to the register
    */
    void i2cWriteByte(char reg, char value);

    /**
    * Read value from register of device
    * @param reg  The device register to read
    * @return     The result
    */
    char i2cReadByte(char reg);
    
    /**
    * Initialize the device
    *
    * @param im DRV2667::Digital or DRV2667::Analog
    * @param gn gain for the amplifier
    * @param to timeout for FIFO interface (digital input only)
    */
    void init(InputMux im = Digital, Gain gn = GNx1, Timeout to = TOx1);

    /**
    * Reset the device
    */
    void reset();
    
    /**
    * Starts waveform playback
    */
    void play();

    /**
    * Cancel waveform playback
    */
    void stop();
    
    /**
    * Get state of playing
    */
    bool isPlaying();
    
    /**
    * Set sequencer to play waveform
    *   
    * @param id array of waveform ID in RAM
    * @param setNum the number of waveform to set
    */
    void setWaveform(char* id, char setNum);

    ///**
    //* Entry point for FIFO data read out automatically at an 8-kHz sampling rate
    //*
    //* @param data signed 8-bit data
    //* @param size size of waveform
    //* To be implemented
    //*/
    //void loadFIFO(signed char* data, char size);

    /**
    * Set internal wavefrom storage for the Waveform Synthesis Playback mode
    * @param data[][0] Peak voltage = amp / 255  x Gain / 2
    * @param data[][1] Sinusoidal frequency
    * @param data[][2] Number of cycles to be played
    * @param data[][3] The envelope setting: bits [7:4] sets ramp-up rate; 
    *                  bits [3:0] sets ramp-down rate. 0x00 NoEnvelope; 0x01 
    *                  32ms; 0x02 64ms; 0x03 96ms; 0x04 128ms; 0x05 160ms; 
    *                  0x06 192ms; 0x07 224ms; 0x08 256ms; 0x09 512ms; 0x0A
    *                  768ms; 0x0B 1024ms; 0xC 1280ms; 0x0D 1536ms; 0x0E
    *                  1792ms; 0x0F 2048ms.
    * @param data[][4] Number of repeat. Setting "0" plays infinite loop until
    *                  the GO bit is cleared by the user.
    * @param waveNum the number of waveform
    */
    void setWSP(char data[][5], char waveNum);

    ///**
    //* Set internal wavefrom storage for the Direct Playback from RAM mode
    //* To be implemented
    //*/
    //void setDPR(void);

private:
    I2C *_i2c;
};

#endif