#ifndef DRV2605_H
#define DRV2605_H

#include "mbed.h"

// DRV2605 Slave Address
#define DRV2605_SLAVE_ADDR 0x5A

// DRV2605 Registers
#define STATUS                               0x00
#define MODE                                 0x01
#define REAL_TIME_PLAYBACK                   0x02
#define LIBRARY_SELECTION                    0x03
#define WAVEFORM_SEQUENCER_1                 0x04
#define WAVEFORM_SEQUENCER_2                 0x05
#define WAVEFORM_SEQUENCER_3                 0x06
#define WAVEFORM_SEQUENCER_4                 0x07
#define WAVEFORM_SEQUENCER_5                 0x08
#define WAVEFORM_SEQUENCER_6                 0x09
#define WAVEFORM_SEQUENCER_7                 0x0A
#define WAVEFORM_SEQUENCER_8                 0x0B
#define GO                                   0x0C
#define OVERDRIVE_TIME_OFFSET                0x0D
#define POSITIVE_SUSTAIN_TIME_OFFSET         0x0E
#define NEGATIVE_SUSTAIN_TIME_OFFSET         0x0F
#define BRAKE_TIME_OFFSET                    0x10
#define AUDIO_TO_VIBE_CONTROL                0x11
#define AUDIO_TO_VIBE_MINIMUM_INPUT_LEVEL    0x12
#define AUDIO_TO_VIBE_MAXIMUM_INPUT_LEVEL    0x13
#define AUDIO_TO_VIBE_MINIMUM_OUPUT_DRIVE    0x14
#define AUDIO_TO_VIBE_MAXIMUM_OUTPUT_DRIVE   0x15
#define RATED_VOLTAGE                        0x16
#define OVERDRIVE_CLAMP_VOLTAGE              0x17
#define AUTO_CALIBRATION_COMPENSATION_RESULT 0x18
#define AUTO_CALIBRATION_BACK_EMF_RESULT     0x19
#define FEEDBACK_CONTROL                     0x1A
#define CONTROL1                             0x1B
#define CONTROL2                             0x1C
#define CONTROL3                             0x1D
#define CONTROL4                             0x1E
#define CONTROL5                             0x1F
#define LRA_OPEN_LOOP_PERIOD                 0x20
#define VBAT_VOLTAGE_MONITOR                 0x21
#define LRA_RESONANCE_PERIOD                 0x22

// Modes defines the possible modes of the DRV2605L (Register Addr: 0x01)
#define Mode_INTERNAL_TRIG   0x00 // Waveforms fired by Setting GO bit in Register 0x0C
#define Mode_EXTERNAL_EDGE   0x01 // Rising Edge on IN/TRIG pin set GO Bit.
#define Mode_EXTERNAL_LEVEL  0x02 // GO bit follows state of edge on IN/TRIG pin.
#define Mode_PWM_ANALOG      0x03 // PWM or Analog Signal accepted at IN/TRIG pin.
#define Mode_AUDIO_TO_VIBE   0x04 // An AC-coupled audio signal is accepted at the IN/TRIG pin.
#define Mode_RTP             0x05 // Real- Time Playback
#define Mode_DIAG            0x06 // Set to perform actuator diagnostics
#define Mode_AUTO_CAL        0x07 // Set to perform auto calibration of device for actuator
#define Mode_STANDBY         0x40 // Set Device to Software Standby (Low- Power Mode)
#define Mode_RESET           0x80 // Reset Device (equivalent of power cycling the device)

// FeedBack_Controls Fields Bitmasks (Register Addr: 0x1A)

// ROM Waveform Library Selections
#define LibraryEMPTY            0x00
#define LibraryA                0x01
#define LibraryB                0x02
#define LibraryC                0x03
#define LibraryD                0x04
#define LibraryE                0x05
#define LibraryLRA              0x06
#define LibraryF                0x07

/** Library for the TI DRV2605
 *
 */

class DRV2605
{

public:
    // FeedBack_Controls Fields Bitmasks
    enum ActuatorType {ERM = 0, LRA = 0x80};
    enum BrakeFactor {x1 = 0x00, x2 = 0x10, x3 = 0x20, x4 = 0x30, x6 = 0x40, x8 = 0x50, x16 = 0x60, DISABLE = 0x70};
    enum LoopGain {LOW = 0x00, MEDIUM = 0x04, HIGH = 0x08, VERYHIGH = 0x0C};

    // Control2 Fields Bitmasks
    enum SampleTime {us150 = 0x00, us200 = 0x10, us250 = 0x20, us300 = 0x30};

    // Control4 Fields Bitmasks
    enum AutoCalTime {ms150_350 = 0x00, ms250_450 = 0x10, ms500_700 = 0x20, ms1000_1200 = 0x30};

    /**
    * Create a DRV2605 object
    *
    * @param &i2c pointer of I2C object
    * @param at actuator type
    * @param bf brake factor
    * @param lg loop gain
    * @param rv rated voltage
    * @param ocv overdirve clamp voltage
    * @param act auto caliblration time
    * @param dt drive time
    * @param st sample time
    * @param bt braking time
    * @param it current dissipation time
    */
    DRV2605(I2C &i2c, ActuatorType at = ERM, BrakeFactor bf = x4,
            LoopGain lg = MEDIUM, uint8_t rv = 0x3F, uint8_t ocv = 0x89,
            AutoCalTime act = ms500_700, uint8_t dt = 0x13,
            SampleTime st = us200, uint8_t bt = 2, uint8_t it = 2);

    /**
    * Write value to register
    *
    * @param reg Register to write
    * @param value Value to write to the register
    */
    void i2cWriteByte(char reg, char value);

    /**
    * Read value from register
    *
    * @param reg Register to read
    * @return Value
    */
    uint8_t i2cReadByte(char reg);

    /**
    * Set the device into specified mode
    *
    * @param mode Mode to set
    */
    void mode(uint8_t mode);

    /**
    * Runs diagnostics
    *
    * @return Results of the diagnostics
    */
    uint8_t diagnostics();

    /**
    * Select the input data interpretation for RTP
    *
    * @param isSigned true: singed(-127~127), false: unsigned (0~255)
    */
    void dataFormatRTP(bool isSigned);

    /**
    * Set ROM Library
    *
    * Note: This should be called before setWaveform function
    * @param lib Set library (see library's macro)
    */
    void setLibrary(uint8_t lib);

    /**
    * Set single waveform from ROM Library
    *
    * Note: This should be called after setLibrary function
    * @param effect The Waveform Effect Library Index value to play (1 to 123)
    */
    void setWaveform(int effect);

    /**
    * Set waveform Sequence
    *
    * Note: This should be called after setLibrary function
    * @param effect1... effect8 The Waveform Effect Library Index value to play
    * (1 to 123): 0 = Stop Condition; 1- 123 = Waveform Index
    */
    void setWaveformSequence(int effect1 = 0, int effect2 = 0,
                             int effect3 = 0, int effect4 = 0,
                             int effect5 = 0, int effect6 = 0,
                             int effect7 = 0, int effect8 = 0);

    /**
    *    Plays the currently loaded waveform or waveform sequence.
    */
    void play();

    /**
    *    Stops the currently loaded waveform or waveform sequence.
    */
    void stop();

    /**
    * Real time play back
    *
    * @param rtp  intensity of the vibration (0 - 255)
    */
    void playRtp(uint8_t rtp);

    /**
    * Battery voltage monitor
    *
    * @return battery voltage
    */
    float batteryVoltage(void);

private:
    I2C *_i2c;
};

#endif