/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@brief Doxygen documentation.
@brief Wheather Station.
@author Panayiotis Rodosthenous
@date   March 2015
*/

#ifndef MAIN_H
#define MAIN_H
 
#define PI 3.14159265359
 
#include "mbed.h"
#include "N5110.h"
#include "BMP180.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"
#include "beep.h"

/**  
@namespace bmp180
@brief Object to ssign sensor pins to the mbed
*/
BMP180 bmp180(p28,p27);   // SDA, SCL

/**  
@namespace serial
@brief Serial connection for presenting data to pc used for debugging
*/
Serial serial(USBTX,USBRX); //tx,rx

/**  
@namespace pc
@brief Serial connection for presenting data to serial
*/
Serial pc(USBTX,USBRX); //tx,rx

/**  
@namespace lcd
@brief Object to assign the LCD pins to the mbed
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);    ///VCC,SCE,RST,D/C,MOSI,SCLK,LED

/**  
@namespace local
@brief Location to create a file for data logging
*/
LocalFileSystem local("local"); 

/**  
@namespace buzzer
@brief GPIO output for warning buzzer 
*/
Beep buzzer(p21);

/**  
@namespace button4
@brief GPIO input for main screen button
*/
DigitalIn button4(p19); 

/**  
@namespace button
@brief ISR input for temperature graph button
*/
InterruptIn button(p18);  

/**  
@namespace button2
@brief ISR input for pressure graph button
*/
InterruptIn button2(p17);  
  
/**  
@namespace button3
@brief ISR input for changing the units button
*/
InterruptIn button3(p16);

/**  
@namespace switch_logging
@brief GPIO input for data logging switch
*/
DigitalIn switch_logging(p15);

/**  
@namespace ain
@brief GPIO analogue input potentiometer for controlling the brightness of the LCD
*/
AnalogIn ain(p20);   

/**  
@namespace green
@brief GPIO output for status green LED
*/
DigitalOut green(p25);

/**  
@namespace red
@brief GPIO output for status red LED
*/
DigitalOut red(p24);

/**  
@namespace leds
@brief Bus output for status on-board LED
*/
BusOut leds(LED4,LED3,LED2,LED1);   //bus for on-board LEDs

/**  
@namespace timerReading
@brief Ticker for reading the sensor
*/
Ticker timerReading;

/**  
@namespace timer
@brief Ticker for getting current time
*/
Ticker timer;

/**  
@namespace timerFiles
@brief Ticker for writing data to file
*/
Ticker timerFiles;  

/**  
@namespace timerClearScreen
@brief Ticker for clearing the LCD
*/
Ticker timerClearScreen;

int readingFlag = 0;  /*!< reading flag set in ISR */
int timerFlag = 0;  /*!< timer flag set in ISR */
int timerFilesFlag = 0; /*!< timerFiles flag set in ISR */
int setTimeFlag = 0; /*!< setTime flag set in ISR */
int graphTempFlag = 0;  /*!< graphTemp flag set in ISR */
int graphPresFlag = 0;  /*!< graphPres flag set in ISR */
int unitsFlag = 0;  /*!< units flag set in ISR */
int clearFlag = 0; /*!< celar flag set in ISR */
int i = 0;  /*!< variable i used as index in print() */

char rxString[16];  /*!< string used to transfer current time into the buffer */
char bufferT[14];   /*!< buffer used to store current time */
char bufferD[14];   /*!< buffer used to store current date */

float temperature;  /*!< variable used to store current temperature */
float pressure;     /*!< variable used to store current pressure */
float array[84];    /*!< array used to store temperature readings */
float array2[84];   /*!< array used to store pressure readings */

/** Update
    *
    *   Prints and updates the thermometer presented on the main screen.
    *   For each specified range of temperatures adds or removes pixels from inside the thermometer.
    *   Introduces a live update on the main screen.
    */
void update();

/** Print
    *
    *   Prints the different screens on the LCD.
    *   According to the buttons status-flags, a different screen is presented. Buttons are used as interrupts.
    *   Includes the update() function.
    *   -Button 1: Print the temperature graph with the average temperature and weather prediction. Disables button 2.
    *   -Button 2: Print the pressure graph with the average pressure and weather prediction. Disables button 1.
    *   -Button 3: Converts the temperature and pressure units on the main screen. When one of buttons 1 or 2 is previously pressed, it removes the average and weather predictions from screen.
    *   -No button pressed: Displays the main screen.
    */
void print();

/** Write Data to File
    *
    *   Logs the time, temperature and pressure on a .csv file on the mbed Flash memory.
    *   It is called when the logging switch is ON.
    *   A ticker is used to call this function. Logs data every 10 sec.
    *   @param  time - the current time taken from setTime() function.
    *   @param  temperature - the current temperature read from sensor.
    *   @param  temperature - the current pressure read from sensor.
    *   @returns
    *       log.csv - file with parameters in comma separated format stored on mbed. 
    */
void writeDataToFile(float tempereture,char* time,float pressure);

/** Serial ISR
    *
    *   This is an ISR function, called when serial data is received.
    *   Gets the time rxString.
    */
void serialISR(); 

/** Set Time
    *
    *   This function sets the UNIX time.
    *   Updates the time, bu converting the time string into an integer value.
    */
void setTime();

/** Timer Expired Reading
    *
    *   This function sets the reading flag high when called.
    *   Attached to the timerReading ticker.
    */
void timerExpiredReading();

/** Timer Expired Time
    *
    *   This function sets the timer flag high when called.
    *   Attached to the timer ticker.
    */
void timerExpiredTime();

/** Timer Expired Files
    *
    *   This function sets the timerFiles flag high when called.
    *   Attached to the timerFiles ticker.
    */
void timeExpiredFiles();

/** Timer Expired Clear
    *
    *   This function sets the clear flag high when called.
    *   Attached to the clearScreen ticker.
    */
void timerExpiredClear();

/** Button for Temperature Pressed
    *
    *   This function toggles the graphTemp flag when button is pressed.
    *   Attached to the button interrupt configuration.
    */
void buttonTempPressed();

/** Button for Pressure Pressed
    *
    *   This function toggles the graphPres flag when button2 is pressed.
    *   Attached to the button2 interrupt configuration.
    */
void buttonPresPressed();

/** Button for Units Pressed
    *
    *   This function toggles the units flag when button3 is pressed.
    *   Attached to the button3 interrupt configuration.
    */
void buttonUnitsPressed();

#endif