#ifndef USE_CAN_F303K8
#include "use_can_stm.h"

int CanStm::init(int baudrate){
    int result;
    CAN_TIMEOUT_VALUE = 3.0;
    timer.start();

    switch(baudrate)
    {
        case CAN_125KHZ:
            canHandler.Init.Prescaler = 16;
            break;
        
        case CAN_250KHZ:    
            canHandler.Init.Prescaler = 8;
            break;
            
        case CAN_500KHZ:
            canHandler.Init.Prescaler = 4;
            break;
            
        case CAN_1MHZ:
            canHandler.Init.Prescaler = 2;
            break;
    }
    
    canHandler.Instance = CAN1;                     
    canHandler.Init.Mode = CAN_MODE_NORMAL;        
    canHandler.Init.SJW = CAN_SJW_1TQ;  
    canHandler.Init.BS1 = CAN_BS1_11TQ; 
    canHandler.Init.BS2 = CAN_BS2_4TQ;  
    canHandler.Init.TTCM = DISABLE;                 //繧ｿ繧､繝�繝医Μ繧ｬ騾壻ｿ｡繝｢繝ｼ繝芽ｨｭ螳�
    canHandler.Init.ABOM = DISABLE;                  //閾ｪ蜍輔ヰ繧ｹ繧ｪ繝慕ｮ｡逅�險ｭ螳�
    canHandler.Init.AWUM = DISABLE;                 //閾ｪ蜍輔え繧ｧ繧､繧ｯ繧｢繝�繝励Δ繝ｼ繝芽ｨｭ螳�
    canHandler.Init.NART = DISABLE;                 //閾ｪ蜍募�埼�∫ｦ∵ｭ｢繝｢繝ｼ繝芽ｨｭ螳�
    canHandler.Init.RFLM = DISABLE;                 //FIFO繝ｭ繝�繧ｯ繝｢繝ｼ繝芽ｨｭ螳�
    canHandler.Init.TXFP = DISABLE;                 //騾∽ｿ｡FIFO縺ｮ蜆ｪ蜈磯��菴崎ｨｭ螳�

    if(CanStm::hal_can_init(&canHandler) == HAL_OK){
        result = SUCCESS;
    } else {
        result = ERROR;
    }

    return result;

}

int CanStm::transmit(uint32_t id, int format, uint8_t* tx_array){

    CanTxMsgTypeDef TxMsg;
    int result;
    canHandler.pTxMsg = &TxMsg;
        
    switch(format){
        case CAN_ID_STD:
            TxMsg.StdId   =id;
            break;
        case CAN_ID_EXT:
            TxMsg.ExtId   =id;
            break;
    }

    TxMsg.RTR     = CAN_RTR_DATA;
    TxMsg.IDE     = format;
    TxMsg.DLC     = 8;
    
    for(int i=0;i<8;i++){
        TxMsg.Data[i] = tx_array[i];
    }
    
    result = (CanStm::hal_can_transmit(&canHandler,0xFF) == HAL_OK) ? 1 : 0;
    return result;
}

int CanStm::attach(uint32_t id,void func(),int irq_mode){
    int result;
    CAN_FilterConfTypeDef conf;

    function = (uint32_t)func;

    switch(irq_mode){
        case RX_FIFO0:
            use_fifo = CAN_FIFO0;
            conf.FilterFIFOAssignment = CAN_FILTER_FIFO0;
            //FIFO0受信割り込み許可
            CanStm::setPriority(CAN_RX0_IRQn);
            __NVIC_EnableIRQ(CAN_RX0_IRQn);
            __NVIC_SetVector(CAN_RX0_IRQn,function);
            SET_BIT(canHandler.Instance->IER,CAN_IER_FMPIE0);
            break;

        case RX_FIFO1:
            use_fifo = CAN_FIFO1;
            conf.FilterFIFOAssignment = CAN_FILTER_FIFO1;
            //FIFO1受信割り込み許可
            CanStm::setPriority(CAN_RX1_IRQn);
            __NVIC_EnableIRQ(CAN_RX1_IRQn);
            __NVIC_SetVector(CAN_RX1_IRQn,function);
            SET_BIT(canHandler.Instance->IER,CAN_IER_FMPIE1);
            break;

        case TX_READY:
            //送信メールボックス空き割り込み許可
            CanStm::setPriority(CAN_TX_IRQn);
            __NVIC_EnableIRQ(CAN_TX_IRQn);
            __NVIC_SetVector(CAN_TX_IRQn,function);
            SET_BIT(canHandler.Instance->IER,CAN_IER_TMEIE);
            break;
    }

    /* setting the filter */
    conf.BankNumber = 0;
    conf.FilterNumber = 0;
    conf.FilterActivation = ENABLE;
    conf.FilterIdHigh = id<<5;
    conf.FilterIdLow = 0;
    conf.FilterMaskIdHigh = 0b11111111111<<5;
    conf.FilterMaskIdLow = 0;
    conf.FilterMode = CAN_FILTERMODE_IDMASK;
    conf.FilterScale = CAN_FILTERSCALE_32BIT;

    if(CanStm::hal_can_configfilter(&canHandler, &conf) == HAL_OK){
        result = SUCCESS;
    } else {
        result = ERROR;
    }
    return result;
}

int CanStm::receive(){
    uint32_t tickstart = 0U;
    uint32_t Timeout = 1.0;
    uint8_t FIFONumber;

    FIFONumber = use_fifo;

    /* Check the parameters */
    assert_param(IS_CAN_FIFO(FIFONumber));

    /* Process locked */
    __HAL_LOCK(&canHandler);

    /* Check if CAN state is not busy for RX FIFO0 */
    if ((FIFONumber == CAN_FIFO0) && ((canHandler.State == CAN_STATE::BUSY_RX0) ||   \
                                (canHandler.State == CAN_STATE::BUSY_TX_RX0) ||      \
                                (canHandler.State == CAN_STATE::BUSY_RX0_RX1) ||     \
                                (canHandler.State == CAN_STATE::BUSY_TX_RX0_RX1)))
    {
        /* Process unlocked */
        __HAL_UNLOCK(&canHandler);

        return BUSY;
    }

    /* Check if CAN state is not busy for RX FIFO1 */
    if ((FIFONumber == CAN_FIFO1) && ((canHandler.State == CAN_STATE::BUSY_RX1) ||   \
                                (canHandler.State == CAN_STATE::BUSY_TX_RX1) ||      \
                                (canHandler.State == CAN_STATE::BUSY_RX0_RX1) ||     \
                                (canHandler.State == CAN_STATE::BUSY_TX_RX0_RX1)))
    {
        /* Process unlocked */
        __HAL_UNLOCK(&canHandler);

        return BUSY;
    }

    /* Change CAN state */
    if (FIFONumber == CAN_FIFO0)
    {
        switch(canHandler.State)
        {
            case CAN_STATE::BUSY_TX:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX0;
                break;
            case CAN_STATE::BUSY_RX1:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX0_RX1;
                break;
            case CAN_STATE::BUSY_TX_RX1:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX0_RX1;
                break;
            default: /* CAN_STATE::READY */
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX0;
                break;
        }
    }
    else /* FIFONumber == CAN_FIFO1 */
        {
        switch(canHandler.State)
        {
            case CAN_STATE::BUSY_TX:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX1;
                break;
            case CAN_STATE::BUSY_RX0:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX0_RX1;
                break;
            case CAN_STATE::BUSY_TX_RX0:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX0_RX1;
                break;
            default: /* CAN_STATE::READY */
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX1;
                break;
        }
    }

    /* Get tick */
    timer.reset();
    tickstart = timer.read();

    /* Check pending message */
    while(__HAL_CAN_MSG_PENDING(&canHandler, FIFONumber) == 0U)
    {
        /* Check for the Timeout */
        if(Timeout != HAL_MAX_DELAY)
        {
            if((Timeout == 0U) || ((timer.read()-tickstart) > Timeout))
            {
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::TIMEOUT;

                /* Process unlocked */
                __HAL_UNLOCK(&canHandler);

                return TIMEOUT;
            }
        }
    }

    /* Set RxMsg pointer */
    if(FIFONumber == CAN_FIFO0){
        CanStm::ConvertMessage(&canHandler,&rx_data[FIFO_0],CAN_FIFO0);
    } else /* FIFONumber == CAN_FIFO1 */ {
        CanStm::ConvertMessage(&canHandler,&rx_data[FIFO_1],CAN_FIFO1);
    }

    /* Release the FIFO */
    if(FIFONumber == CAN_FIFO0)
    {
        /* Release FIFO0 */
        __HAL_CAN_FIFO_RELEASE(&canHandler, CAN_FIFO0);
    }
    else /* FIFONumber == CAN_FIFO1 */
    {
        /* Release FIFO1 */
        __HAL_CAN_FIFO_RELEASE(&canHandler, CAN_FIFO1);
    }

    /* Change CAN state */
    if (FIFONumber == CAN_FIFO0)
    {
        switch(canHandler.State)
        {
            case CAN_STATE::BUSY_TX_RX0:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX;
                break;
            case CAN_STATE::BUSY_RX0_RX1:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX1;
                break;
            case CAN_STATE::BUSY_TX_RX0_RX1:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX1;
                break;
            default: /* (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX0 */
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::READY;
                break;
        }
    }
    else /* FIFONumber == CAN_FIFO1 */
    {
        switch(canHandler.State)
        {
            case CAN_STATE::BUSY_TX_RX1:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX;
                break;
            case CAN_STATE::BUSY_RX0_RX1:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX0;
                break;
            case CAN_STATE::BUSY_TX_RX0_RX1:
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX0;
                break;
            default: /* CAN_STATE::BUSY_RX1 */
                canHandler.State = (HAL_CAN_StateTypeDef)CAN_STATE::READY;
                break;
        }
    }

    /* Process unlocked */
    __HAL_UNLOCK(&canHandler);

    /* Return function status */
    return SUCCESS;
}

int CanStm::getData(int data_category){
    int output;

    switch(data_category){
        case STANDARD_ID:
            output = (int)rx_data[use_fifo].StdId;
            break;
        case EXTENDED_ID:
            output = (int)rx_data[use_fifo].ExtId;
            break;
        case IDE:
            output = (int)rx_data[use_fifo].IDE;
            break;
        case RTR:
            output = (int)rx_data[use_fifo].RTR;
            break;
        case DLC:
            output = (int)rx_data[use_fifo].DLC;
            break;
        case FMI:
            output = (int)rx_data[use_fifo].FMI;
            break;
        case FIFO_NUMBER:
            output = (int)rx_data[use_fifo].FIFONumber;
            break;
        case RX_DATA_0:
            output = (int)rx_data[use_fifo].Data[0];
            break;
        case RX_DATA_1:
            output = (int)rx_data[use_fifo].Data[1];
            break;
        case RX_DATA_2:
            output = (int)rx_data[use_fifo].Data[2];
            break;
        case RX_DATA_3:
            output = (int)rx_data[use_fifo].Data[3];
            break;
        case RX_DATA_4:
            output = (int)rx_data[use_fifo].Data[4];
            break;
        case RX_DATA_5:
            output = (int)rx_data[use_fifo].Data[5];
            break;
        case RX_DATA_6:
            output = (int)rx_data[use_fifo].Data[6];
            break;
        case RX_DATA_7:
            output = (int)rx_data[use_fifo].Data[7];
            break;
        default :
            output = 0;
            break;
    }
    return output;
}

void CanStm::exitIrqHandler(){
    SET_BIT(canHandler.Instance->RF0R,CAN_RF0R_RFOM0);
    SET_BIT(canHandler.Instance->IER,CAN_IER_FMPIE0);
}

HAL_StatusTypeDef CanStm::hal_can_init(CAN_HandleTypeDef* hcan){

      uint32_t status = CAN_INITSTATUS_FAILED;  /* Default init status */
      float tickstart = 0U;
      /* Check CAN handle */
      if(hcan == NULL){
         return HAL_ERROR;
      }

      /* Check the parameters */
      assert_param(IS_CAN_ALL_INSTANCE(hcan->Instance));
      assert_param(IS_FUNCTIONAL_STATE(hcan->Init.TTCM));
      assert_param(IS_FUNCTIONAL_STATE(hcan->Init.ABOM));
      assert_param(IS_FUNCTIONAL_STATE(hcan->Init.AWUM));
      assert_param(IS_FUNCTIONAL_STATE(hcan->Init.NART));
      assert_param(IS_FUNCTIONAL_STATE(hcan->Init.RFLM));
      assert_param(IS_FUNCTIONAL_STATE(hcan->Init.TXFP));
      assert_param(IS_CAN_MODE(hcan->Init.Mode));
      assert_param(IS_CAN_SJW(hcan->Init.SJW));
      assert_param(IS_CAN_BS1(hcan->Init.BS1));
      assert_param(IS_CAN_BS2(hcan->Init.BS2));
      assert_param(IS_CAN_PRESCALER(hcan->Init.Prescaler));

      if(hcan->State == CAN_STATE::RESET){
        /* Allocate lock resource and initialize it */
        hcan->Lock = HAL_UNLOCKED;
        /* Init the low level hardware */
        CanStm::hal_can_mspinit(hcan);
      }

      /* Initialize the CAN state*/
      hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY;

      /* Exit from sleep mode */
      CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_SLEEP);

      /* Request initialisation */
      SET_BIT(hcan->Instance->MCR, CAN_MCR_INRQ);

      /* Get tick */
      tickstart = timer.read();

      /* Wait the acknowledge */
      while(HAL_IS_BIT_CLR(hcan->Instance->MSR, CAN_MSR_INAK)){
          if((timer.read()-tickstart) > CAN_TIMEOUT_VALUE){
              hcan->State= (HAL_CAN_StateTypeDef)CAN_STATE::TIMEOUT;
              /* Process unlocked */
              __HAL_UNLOCK(hcan);
              return HAL_TIMEOUT;
          }
      }

      /* Check acknowledge */
      if (HAL_IS_BIT_SET(hcan->Instance->MSR, CAN_MSR_INAK))
      {
        /* Set the time triggered communication mode */
        if (hcan->Init.TTCM == ENABLE)
        {
          SET_BIT(hcan->Instance->MCR, CAN_MCR_TTCM);
        }
        else
        {
          CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_TTCM);
        }

        /* Set the automatic bus-off management */
        if (hcan->Init.ABOM == ENABLE)
        {
          SET_BIT(hcan->Instance->MCR, CAN_MCR_ABOM);
        }
        else
        {
          CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_ABOM);
        }

        /* Set the automatic wake-up mode */
        if (hcan->Init.AWUM == ENABLE){
          SET_BIT(hcan->Instance->MCR, CAN_MCR_AWUM);
        } else {
          CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_AWUM);
        }

        /* Set the no automatic retransmission */
        if (hcan->Init.NART == ENABLE){
          SET_BIT(hcan->Instance->MCR, CAN_MCR_NART);
        } else {
          CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_NART);
        }

        /* Set the receive FIFO locked mode */
        if (hcan->Init.RFLM == ENABLE){
          SET_BIT(hcan->Instance->MCR, CAN_MCR_RFLM);
        } else {
          CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_RFLM);
        }

        /* Set the transmit FIFO priority */
        if (hcan->Init.TXFP == ENABLE){
          SET_BIT(hcan->Instance->MCR, CAN_MCR_TXFP);
        } else {
          CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_TXFP);
        }

        /* Set the bit timing register */
        WRITE_REG(hcan->Instance->BTR, (uint32_t)(hcan->Init.Mode           |
                                                  hcan->Init.SJW            |
                                                  hcan->Init.BS1            |
                                                  hcan->Init.BS2            |
                                                  (hcan->Init.Prescaler - 1U) ));

        /* Request leave initialisation */
        CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_INRQ);

        /* Get tick */
        tickstart = timer.read();

        /* Wait the acknowledge */
        while(HAL_IS_BIT_SET(hcan->Instance->MSR, CAN_MSR_INAK)){
          if((timer.read()-tickstart) > CAN_TIMEOUT_VALUE){
             hcan->State= (HAL_CAN_StateTypeDef)CAN_STATE::TIMEOUT;

             /* Process unlocked */
           __HAL_UNLOCK(hcan);

           return HAL_TIMEOUT;
          }
        }

        /* Check acknowledged */
        if(HAL_IS_BIT_CLR(hcan->Instance->MSR, CAN_MSR_INAK)){
          status = CAN_INITSTATUS_SUCCESS;
        }
      }

      if(status == CAN_INITSTATUS_SUCCESS){
        /* Set CAN error code to none */
        hcan->ErrorCode = HAL_CAN_ERROR_NONE;

        /* Initialize the CAN state */
        hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::READY;

        /* Return function status */
        return HAL_OK;
      } else {
        /* Initialize the CAN state */
        hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::ERROR;

        /* Return function status */
        return HAL_ERROR;
      }
}

void CanStm::hal_can_mspinit(CAN_HandleTypeDef* hcan){
      GPIO_InitTypeDef GPIO_InitStruct;
      if(hcan->Instance==CAN1){
        /* Peripheral clock enable */
        __HAL_RCC_CAN1_CLK_ENABLE();

        /**CAN GPIO Configuration
        PA11 -> CAN_RX
        PA12 -> CAN_TX
        */
        GPIO_InitStruct.Pin = GPIO_PIN_11|GPIO_PIN_12;
        GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull = GPIO_NOPULL;
        GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
        GPIO_InitStruct.Alternate = GPIO_AF9_CAN;
        HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
      }
}

HAL_StatusTypeDef CanStm::hal_can_transmit(CAN_HandleTypeDef* hcan, uint32_t Timeout){

      uint32_t transmitmailbox = CAN_TXSTATUS_NOMAILBOX;
      uint32_t tickstart = 0U;

      /* Check the parameters */
      assert_param(IS_CAN_IDTYPE(hcan->pTxMsg->IDE));
      assert_param(IS_CAN_RTR(hcan->pTxMsg->RTR));
      assert_param(IS_CAN_DLC(hcan->pTxMsg->DLC));

      if(((hcan->Instance->TSR&CAN_TSR_TME0) == CAN_TSR_TME0) || \
         ((hcan->Instance->TSR&CAN_TSR_TME1) == CAN_TSR_TME1) || \
         ((hcan->Instance->TSR&CAN_TSR_TME2) == CAN_TSR_TME2)){
        /* Process locked */
        __HAL_LOCK(hcan);

        /* Change CAN state */
        switch(hcan->State){
          case CAN_STATE::BUSY_RX0:
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX0;
              break;
          case CAN_STATE::BUSY_RX1:
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX1;
              break;
          case CAN_STATE::BUSY_RX0_RX1:
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX_RX0_RX1;
              break;
          default: /* CAN_STATE::READY */
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_TX;
              break;
        }

        /* Select one empty transmit mailbox */
        if (HAL_IS_BIT_SET(hcan->Instance->TSR, CAN_TSR_TME0)){
          transmitmailbox = CAN_TXMAILBOX_0;
        } else if (HAL_IS_BIT_SET(hcan->Instance->TSR, CAN_TSR_TME1)) {
          transmitmailbox = CAN_TXMAILBOX_1;
        } else {
          transmitmailbox = CAN_TXMAILBOX_2;
        }

        /* Set up the Id */
        hcan->Instance->sTxMailBox[transmitmailbox].TIR &= CAN_TI0R_TXRQ;
        if (hcan->pTxMsg->IDE == CAN_ID_STD) {
          assert_param(IS_CAN_STDID(hcan->pTxMsg->StdId));
          hcan->Instance->sTxMailBox[transmitmailbox].TIR |= ((hcan->pTxMsg->StdId << CAN_TI0R_STID_Pos) | \
                                                               hcan->pTxMsg->RTR);
        } else {
          assert_param(IS_CAN_EXTID(hcan->pTxMsg->ExtId));
          hcan->Instance->sTxMailBox[transmitmailbox].TIR |= ((hcan->pTxMsg->ExtId << CAN_TI0R_EXID_Pos) | \
                                                               hcan->pTxMsg->IDE | \
                                                               hcan->pTxMsg->RTR);
        }

        /* Set up the DLC */
        hcan->pTxMsg->DLC &= (uint8_t)0x0000000FU;
        hcan->Instance->sTxMailBox[transmitmailbox].TDTR &= 0xFFFFFFF0U;
        hcan->Instance->sTxMailBox[transmitmailbox].TDTR |= hcan->pTxMsg->DLC;

        /* Set up the data field */
        WRITE_REG(hcan->Instance->sTxMailBox[transmitmailbox].TDLR, ((uint32_t)hcan->pTxMsg->Data[3] << CAN_TDL0R_DATA3_Pos) |
                                                                    ((uint32_t)hcan->pTxMsg->Data[2] << CAN_TDL0R_DATA2_Pos) |
                                                                    ((uint32_t)hcan->pTxMsg->Data[1] << CAN_TDL0R_DATA1_Pos) |
                                                                    ((uint32_t)hcan->pTxMsg->Data[0] << CAN_TDL0R_DATA0_Pos));
        WRITE_REG(hcan->Instance->sTxMailBox[transmitmailbox].TDHR, ((uint32_t)hcan->pTxMsg->Data[7] << CAN_TDL0R_DATA3_Pos) |
                                                                    ((uint32_t)hcan->pTxMsg->Data[6] << CAN_TDL0R_DATA2_Pos) |
                                                                    ((uint32_t)hcan->pTxMsg->Data[5] << CAN_TDL0R_DATA1_Pos) |
                                                                    ((uint32_t)hcan->pTxMsg->Data[4] << CAN_TDL0R_DATA0_Pos));

        /* Request transmission */
        SET_BIT(hcan->Instance->sTxMailBox[transmitmailbox].TIR, CAN_TI0R_TXRQ);

        /* Get tick */
        tickstart = HAL_GetTick();

        /* Check End of transmission flag */
        while(!(__HAL_CAN_TRANSMIT_STATUS(hcan, transmitmailbox))) {
          /* Check for the Timeout */
          if(Timeout != HAL_MAX_DELAY) {
            if((Timeout == 0U) || ((HAL_GetTick()-tickstart) > Timeout)) {
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::TIMEOUT;

              /* Cancel transmission */
              __HAL_CAN_CANCEL_TRANSMIT(hcan, transmitmailbox);

              /* Process unlocked */
              __HAL_UNLOCK(hcan);
              return HAL_TIMEOUT;
            }
          }
        }

        /* Change CAN state */
        switch(hcan->State) {
          case CAN_STATE::BUSY_TX_RX0:
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX0;
              break;
          case CAN_STATE::BUSY_TX_RX1:
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX1;
              break;
          case CAN_STATE::BUSY_TX_RX0_RX1:
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::BUSY_RX0_RX1;
              break;
          default: /* CAN_STATE::BUSY_TX */
              hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::READY;
              break;
        }

        /* Process unlocked */
        __HAL_UNLOCK(hcan);

        /* Return function status */
        return HAL_OK;
      } else {
        /* Change CAN state */
        hcan->State = (HAL_CAN_StateTypeDef)CAN_STATE::ERROR;

        /* Return function status */
        return HAL_ERROR;
      }
}

HAL_StatusTypeDef CanStm::hal_can_configfilter(CAN_HandleTypeDef* hcan, CAN_FilterConfTypeDef* sFilterConfig){
      uint32_t filternbrbitpos = 0U;

      /* Check the parameters */
      assert_param(IS_CAN_FILTER_NUMBER(sFilterConfig->FilterNumber));
      assert_param(IS_CAN_FILTER_MODE(sFilterConfig->FilterMode));
      assert_param(IS_CAN_FILTER_SCALE(sFilterConfig->FilterScale));
      assert_param(IS_CAN_FILTER_FIFO(sFilterConfig->FilterFIFOAssignment));
      assert_param(IS_FUNCTIONAL_STATE(sFilterConfig->FilterActivation));

      filternbrbitpos = (1U) << sFilterConfig->FilterNumber;

      /* Initialisation mode for the filter */
      SET_BIT(hcan->Instance->FMR, CAN_FMR_FINIT);

      /* Filter Deactivation */
      CLEAR_BIT(hcan->Instance->FA1R, filternbrbitpos);

      /* Filter Scale */
      if (sFilterConfig->FilterScale == CAN_FILTERSCALE_16BIT){
        /* 16-bit scale for the filter */
        CLEAR_BIT(hcan->Instance->FS1R, filternbrbitpos);

        /* First 16-bit identifier and First 16-bit mask */
        /* Or First 16-bit identifier and Second 16-bit identifier */
        hcan->Instance->sFilterRegister[sFilterConfig->FilterNumber].FR1 =
           ((0x0000FFFFU & (uint32_t)sFilterConfig->FilterMaskIdLow) << 16U) |
            (0x0000FFFFU & (uint32_t)sFilterConfig->FilterIdLow);

        /* Second 16-bit identifier and Second 16-bit mask */
        /* Or Third 16-bit identifier and Fourth 16-bit identifier */
        hcan->Instance->sFilterRegister[sFilterConfig->FilterNumber].FR2 =
           ((0x0000FFFFU & (uint32_t)sFilterConfig->FilterMaskIdHigh) << 16U) |
            (0x0000FFFFU & (uint32_t)sFilterConfig->FilterIdHigh);
      }

      if (sFilterConfig->FilterScale == CAN_FILTERSCALE_32BIT){
        /* 32-bit scale for the filter */
        SET_BIT(hcan->Instance->FS1R, filternbrbitpos);

        /* 32-bit identifier or First 32-bit identifier */
        hcan->Instance->sFilterRegister[sFilterConfig->FilterNumber].FR1 =
           ((0x0000FFFFU & (uint32_t)sFilterConfig->FilterIdHigh) << 16U) |
            (0x0000FFFFU & (uint32_t)sFilterConfig->FilterIdLow);

        /* 32-bit mask or Second 32-bit identifier */
        hcan->Instance->sFilterRegister[sFilterConfig->FilterNumber].FR2 =
           ((0x0000FFFFU & (uint32_t)sFilterConfig->FilterMaskIdHigh) << 16U) |
            (0x0000FFFFU & (uint32_t)sFilterConfig->FilterMaskIdLow);
      }

      /* Filter Mode */
      if (sFilterConfig->FilterMode == CAN_FILTERMODE_IDMASK)
      {
        /*Id/Mask mode for the filter*/
        CLEAR_BIT(hcan->Instance->FM1R, filternbrbitpos);
      }
      else /* CAN_FilterInitStruct->CAN_FilterMode == CAN_FilterMode_IdList */
      {
        /*Identifier list mode for the filter*/
        SET_BIT(hcan->Instance->FM1R, filternbrbitpos);
      }

      /* Filter FIFO assignment */
      if (sFilterConfig->FilterFIFOAssignment == CAN_FILTER_FIFO0)
      {
        /* FIFO 0 assignation for the filter */
        CLEAR_BIT(hcan->Instance->FFA1R, filternbrbitpos);
      }
      else
      {
        /* FIFO 1 assignation for the filter */
        SET_BIT(hcan->Instance->FFA1R, filternbrbitpos);
      }

      /* Filter activation */
      if (sFilterConfig->FilterActivation == ENABLE)
      {
        SET_BIT(hcan->Instance->FA1R, filternbrbitpos);
      }

      /* Leave the initialisation mode for the filter */
      CLEAR_BIT(hcan->Instance->FMR, ((uint32_t)CAN_FMR_FINIT));

      /* Return function status */
      return HAL_OK;
}

void CanStm::setPriority(IRQn_Type IRQn)
{
      uint32_t prioritygroup = 0x00U;
      prioritygroup = NVIC_GetPriorityGrouping();

      __NVIC_SetPriority(IRQn, NVIC_EncodePriority(prioritygroup, 0, 0));
}

void CanStm::ConvertMessage(CAN_HandleTypeDef* hcan,CanRxMsgTypeDef* rx_message,uint32_t FIFONumber){
    /* Get the Id */
    rx_message->IDE = CAN_RI0R_IDE & hcan->Instance->sFIFOMailBox[FIFONumber].RIR;
    if (rx_message->IDE == CAN_ID_STD)
    {
        rx_message->StdId = (CAN_RI0R_STID & hcan->Instance->sFIFOMailBox[FIFONumber].RIR) >> CAN_TI0R_STID_Pos;
    }
    else
    {
        rx_message->ExtId = (0xFFFFFFF8U & hcan->Instance->sFIFOMailBox[FIFONumber].RIR) >> CAN_RI0R_EXID_Pos;
    }
    rx_message->RTR = (CAN_RI0R_RTR & hcan->Instance->sFIFOMailBox[FIFONumber].RIR) >> CAN_RI0R_RTR_Pos;
    /* Get the DLC */
    rx_message->DLC = (CAN_RDT0R_DLC & hcan->Instance->sFIFOMailBox[FIFONumber].RDTR) >> CAN_RDT0R_DLC_Pos;
    /* Get the FMI */
    rx_message->FMI = (CAN_RDT0R_FMI & hcan->Instance->sFIFOMailBox[FIFONumber].RDTR) >> CAN_RDT0R_FMI_Pos;
    /* Get the FIFONumber */
    rx_message->FIFONumber = FIFONumber;
    /* Get the data field */
    rx_message->Data[0] = (CAN_RDL0R_DATA0 & hcan->Instance->sFIFOMailBox[FIFONumber].RDLR) >> CAN_RDL0R_DATA0_Pos;
    rx_message->Data[1] = (CAN_RDL0R_DATA1 & hcan->Instance->sFIFOMailBox[FIFONumber].RDLR) >> CAN_RDL0R_DATA1_Pos;
    rx_message->Data[2] = (CAN_RDL0R_DATA2 & hcan->Instance->sFIFOMailBox[FIFONumber].RDLR) >> CAN_RDL0R_DATA2_Pos;
    rx_message->Data[3] = (CAN_RDL0R_DATA3 & hcan->Instance->sFIFOMailBox[FIFONumber].RDLR) >> CAN_RDL0R_DATA3_Pos;
    rx_message->Data[4] = (CAN_RDH0R_DATA4 & hcan->Instance->sFIFOMailBox[FIFONumber].RDHR) >> CAN_RDH0R_DATA4_Pos;
    rx_message->Data[5] = (CAN_RDH0R_DATA5 & hcan->Instance->sFIFOMailBox[FIFONumber].RDHR) >> CAN_RDH0R_DATA5_Pos;
    rx_message->Data[6] = (CAN_RDH0R_DATA6 & hcan->Instance->sFIFOMailBox[FIFONumber].RDHR) >> CAN_RDH0R_DATA6_Pos;
    rx_message->Data[7] = (CAN_RDH0R_DATA7 & hcan->Instance->sFIFOMailBox[FIFONumber].RDHR) >> CAN_RDH0R_DATA7_Pos;
}



#endif
