/*

*/

#include "string.h"

#include "mbed.h"
#include "nrf51_rtc.h"
#include "AS3935_ext.h"
#include <string>

#define NEED_CONSOLE_OUTPUT 1 // Set this if you need debug messages on the console; 
#define NEED_BLE_CONSOLE 1

#if NEED_CONSOLE_OUTPUT
Serial  pc(USBTX, USBRX);
#define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

#if NEED_BLE_CONSOLE
#include "BLE.h"
#include "UARTService.h"
#define BLEC(...) { char __blecstr[32]; sprintf(__blecstr,__VA_ARGS__); if (uart) uart->write(__blecstr, strlen(__blecstr)); }
#else
#define BLEC(...) /* nothing */
#endif /* #if NEED_BLE_CONSOLE */

#if NEED_BLE_CONSOLE
// Prepare BLE device
const uint8_t DEVICE_NAME[] = "BLE Ligttning Sensor";
#endif // #if NEED_BLE_CONSOLE


// Prepare LED device
DigitalOut led1(LED1);
DigitalOut led2(LED2);

AS3935_ext Lightning(I2C_SDA0,I2C_SCL0,0x00,P0_23);
InterruptIn IntLightning(P0_23); //IRQ AS3935

// used for the example only, not required for rtc use
DigitalIn  button1(BUTTON1);  // used to trigger the time report
InterruptIn button1Press(BUTTON1);

// event record buffer struct
const int s_evrecord=10;
struct t_evrecord {
    public:
        int event;
        time_t time;
        int distance;
        
        t_evrecord() {
            event = 0;
            time = 0;
            distance = 0;
        }
        
        char * events() {
            if ( event & 0x8 ) {
                return "Lightning interval";
            } else if ( event & 0x4 ) {
                return "Disturber deteced";
            } else if ( event & 0x1 ) {
                return "Noise level too high";
            } else {
                return "Unknown event";
            }
        }
} evrecord[s_evrecord];

time_t example_time() {
    // set an intial time
    //  ...not really necessary for this example, but it beats setting it to 0 or some non-obvious large integer (# of seconds since 1/1/1970)
    time_t rawtime=0;

    struct tm * init_timeinfo;

    // initialize time
    init_timeinfo = localtime(&rawtime); // note:  must initialize the struct with this before trying to set components
                                         // ...else code goes into the weeds!!
    init_timeinfo->tm_sec = 0;
    init_timeinfo->tm_min = 0;
    init_timeinfo->tm_hour = 0;
    init_timeinfo->tm_mon = 0;
    init_timeinfo->tm_mday = 1;
    init_timeinfo->tm_year = 70;        

    char date[24];
    strftime(date,sizeof(date),"%Y/%m/%d %H:%M:%S ",init_timeinfo);
    DEBUG("Initial time set as %s.\r\n",date);
    
    // compute the proper value for time in time_t type
    rawtime = mktime(init_timeinfo);
    return rawtime;
}

void print_time() {
    // called when a button is pushed, this prints the current time to the USB-connected console
    
    time_t rawtime=rtc.time();
    
    // massage the time into a human-friendly format for printing
    struct tm * timeinfo;
    timeinfo = localtime(&rawtime);
    char date[24];
    strftime(date,sizeof(date),"%Y/%m/%d %H:%M:%S ",timeinfo);
    DEBUG("The current time is %s.(%d)\r\n",date,rawtime);
}

void periodic_update() {
    // for use as interrupt routine, to insure that RTC is updated periodically
    //  ...if rtc is not read before the underlying counter rolls over (typically 512 seconds), the RTC value will be wrong
    //  ...ideally this would be done as part of the nrf51_rtc method, but I couldn't get it to behave (see nrf51_rtc.cpp for details)
    rtc.time();
    Lightning.lightningDistanceKm();
    led1 = !led1;

    // print_time();
}

#if NEED_BLE_CONSOLE
BLEDevice  ble;
UARTService *uart;
static Gap::ConnectionParams_t connectionParams;
int buff_flash_flag;

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)    // Mod
{
    DEBUG("Disconnected handle %u!\n\r", handle);
    DEBUG("Restarting the advertising process\n\r");
    led2 = 0;
    ble.gap().startAdvertising();
    
    
}

void onConnectionCallback(const Gap::ConnectionCallbackParams_t *params)
{
    DEBUG("connected. Got handle %u\r\n", params->handle);

    connectionParams.slaveLatency = 1;
    led2 = 1;
    if (ble.gap().updateConnectionParams(params->handle, &connectionParams) != BLE_ERROR_NONE) {
        DEBUG("failed to update connection paramter\r\n");
    }
    buff_flash_flag = 1;

}

void onDataWritten(const GattWriteCallbackParams *params)
{
    char buff[24];
    int v;
    int afe_gb, nf_lev, wdth, srej;

    int write_param=0;

    if ((uart != NULL) && (params->handle == uart->getTXCharacteristicHandle())) {
        uint16_t bytesRead = params->len;
        strncpy((char*)buff,(char*)params->data,bytesRead);
        buff[bytesRead] = 0x00;
        
        DEBUG("received %u bytes. String '%s'\n\r", bytesRead,buff);
        // ble.updateCharacteristicValue(uart->getRXCharacteristicHandle(), params->data, bytesRead);
        // ble.updateCharacteristicValue(uart->getRXCharacteristicHandle(), buff, bytesRead);
        
        if ( strncmp("get time",buff,bytesRead) == 0 ) {
            char date[30];

            time_t rawtime=rtc.time();
            struct tm * timeinfo= localtime(&rawtime);
            strftime(date,sizeof(date),"%Y/%m/%d %H:%M:%S",timeinfo);
            DEBUG("Current Time:%d %s\r\n",rawtime,date);

            strftime(date,sizeof(date),"%Y%m%d%H%M%S",timeinfo);
            BLEC("%s\n",date);

        } else if ( strncmp("get record ",buff,strlen("get record ")) == 0 ) {
            int i=atoi(&buff[strlen("get record ")]);
            if ( i >= 0 && i < s_evrecord ) {
                char date[24];
                struct tm * timeinfo = localtime(&evrecord[i].time);
                strftime(date,sizeof(date),"%Y/%m/%d %H:%M:%S",timeinfo);
                DEBUG("record %d at %s by %s (%dkm).\r\n",i,date,evrecord[i].events(),evrecord[i].distance);
                BLEC("r%1d:%08x:%1x:%02x\n",i,evrecord[i].time,evrecord[i].event,evrecord[i].distance);
            }
        } else if ( strncmp("st ",buff,strlen("st ")) == 0 ) {
            struct tm * init_timeinfo;
            time_t rawtime;

            // initialize time
            init_timeinfo = localtime(&rawtime); // note:  must initialize the struct with this before trying to set components
                                         // ...else code goes into the weeds!!
            sscanf(buff,"st %4d%2d%2d%2d%2d%2d",&init_timeinfo->tm_year,&init_timeinfo->tm_mon,&init_timeinfo->tm_mday,&init_timeinfo->tm_hour,&init_timeinfo->tm_min,&init_timeinfo->tm_sec);
            init_timeinfo->tm_year -= 1900;
            init_timeinfo->tm_mon --;

            char date[24];
            strftime(date,sizeof(date),"%Y/%m/%d %H:%M:%S",init_timeinfo);
            DEBUG("time set is %s.\r\n",date);

            rawtime = mktime(init_timeinfo);

            rtc.set_time(rawtime);
        } else if ( strncmp("AFE_GB 0x",buff,strlen("AFE_GB 0x")) == 0 ) {
            sscanf(buff,"AFE_GB 0x%2x",&v);
            Lightning.registerWrite(AS3935_AFE_GB,v);
            write_param = 1;
        } else if ( strncmp("NF_LEV 0x",buff,strlen("NF_LEV 0x")) == 0 ) {
            sscanf(buff,"NF_LEV 0x%2x",&v);
            Lightning.registerWrite(AS3935_NF_LEV,v);
            write_param = 1;
        } else if ( strncmp("WDTH 0x",buff,strlen("WDTH 0x")) == 0 ) {
            sscanf(buff,"WDTH 0x%2x",&v);
            Lightning.registerWrite(AS3935_WDTH,v);
            write_param = 1;
        } else if ( strncmp("SREJ 0x",buff,strlen("SREJ 0x")) == 0 ) {
            sscanf(buff,"SREJ 0x%2x",&v);
            Lightning.registerWrite(AS3935_SREJ,v);
            write_param = 1;
        } else if ( strncmp("get param",buff,strlen("get param")) == 0 ) {
            write_param = 1;
        }
        
        if ( write_param != 0 ) {
            afe_gb = Lightning.registerRead(AS3935_AFE_GB);
            nf_lev = Lightning.registerRead(AS3935_NF_LEV);
            wdth   = Lightning.registerRead(AS3935_WDTH);
            srej   = Lightning.registerRead(AS3935_SREJ);
            DEBUG("AFE_GB=0x%02x,NF_LEV=0x%02x,WDTH=0x%02x,SREJ=0x%02x\r\n",afe_gb, nf_lev, wdth, srej);
            BLEC("%02x,%1x,%1x,%1x\n",afe_gb, nf_lev, wdth, srej);
        }
    }
}

#endif // if NEED_BLE_CONSOLE

void DetectLightning()
{
    char OriginInt;
    time_t rawtime=rtc.time();
    struct tm * timeinfo;
    timeinfo = localtime(&rawtime);
    char date[24];// ,sdate[9];
    int distance;
    
    strftime(date,sizeof(date),"%Y/%m/%d %H:%M:%S",timeinfo);
    // strftime(sdate,sizeof(sdate),"%H%M%S",timeinfo);

    wait_ms(2); //on attend 2ms préconisation constructeur
    OriginInt = Lightning.interruptSource();
    distance = Lightning.lightningDistanceKm();

    for(int i=s_evrecord-1;i>0;i--) {
        evrecord[i] = evrecord[i-1];
    }
    evrecord[0].time = rawtime;
    evrecord[0].event = OriginInt;
    evrecord[0].distance = distance;

    if (OriginInt == 1) {
        led2 = !led2; 
        DEBUG("%24s : Noise level too high. %d km\r\n",date,distance);
        }
    if (OriginInt == 4) {
        led2 = !led2;
        DEBUG("%24s : Disturber detected. %d km\r\n",date,distance);
        }
    if (OriginInt == 8) {
        led2 = !led2; 
        DEBUG("%24s : Lightning interrupt %d km\r\n",date,distance);
        }
}


int main(void)
{
#if NEED_BLE_CONSOLE
    ble.init();
    ble.gap().onDisconnection(disconnectionCallback);
    ble.gap().onConnection(onConnectionCallback);
    ble.gattServer().onDataWritten(onDataWritten);
    ble.gap().getPreferredConnectionParams(&connectionParams);
    
    uart = new UARTService(ble);
    buff_flash_flag = 0;

#endif // if NEED_BLE_CONSOLE


    led1=0;
    led2=0;
    int hz=0;

    //initialisations
    wait(1);
    DEBUG("reset\r\n");
    Lightning.reset();
    DEBUG("setTuneCap as 5\r\n");
    Lightning.setTuneCap(5); // Tuning Parameter
    DEBUG("powerup\r\n");
    Lightning.powerUp();

    // AFE Gain (default Inddor:0x12 Outdoor:0x0es)
    DEBUG("set AFE_GB as 0x0d\r\n");
    Lightning.registerWrite(AS3935_AFE_GB,0x0d);

    // Set NF_LEV (default = 2) (0..0x07)
    DEBUG("set Noise Filter as 0x02\r\n");
    Lightning.registerWrite(AS3935_NF_LEV,0x02);
    
    // Set WDTH (default = 2) (0..0x0f)
    DEBUG("set WDTH as 0x02\r\n");
    Lightning.registerWrite(AS3935_WDTH,0x02);

    // Set SREJ
    DEBUG("set SREJ as 0x09\r\n");
    Lightning.registerWrite(AS3935_SREJ,0x09);

    DEBUG("Auto Calibration Start\r\n");
    float minerr = 100;
    int fincap = 7;
    for(int i=0;i<16;i++) {
        Lightning.setTuneCap(i); // Tuning Parameter
        hz = Lightning.MeasureLCOFreq();
        float err = (hz-500000.)/500000.*100.;
        DEBUG("%d : hz=%10d Hz (%5.2f%%)\r\n",i,hz,err);
        if ( abs(err) < minerr ) {
            minerr = abs(err);
            fincap = i;
        }
    }
    Lightning.setTuneCap(fincap); // Tuning Parameter
    wait_ms(100);
    hz = Lightning.MeasureLCOFreq();
    float err = (hz-500000.)/500000.*100.;
    DEBUG("Final %d : hz=%10d Hz (%5.2f%%)\r\n",fincap,hz,err);
    BLEC("%1x:%10dHz:%5.2f:final\n",fincap,hz,err);
    
    DEBUG("Auto Calibration finished\r\n");
    
    // user selectable, any time < 512 seconds is OK
    #define PERIODIC_UPDATE 1
    Ticker rtc_ticker;
    rtc_ticker.attach(&periodic_update, PERIODIC_UPDATE);
    
    time_t initial_time = example_time();
    rtc.set_time(initial_time);
    
    button1Press.fall(&print_time);  // when button1 is pressed, this calls rtc.time() and prints it
    
    IntLightning.rise(&DetectLightning);

#if NEED_BLE_CONSOLE
    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     DEVICE_NAME, sizeof(DEVICE_NAME) - 1);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));

    ble.gap().setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.gap().startAdvertising();
#endif // #if NEED_BLE_CONSOLE

    while (true) {
#if NEED_BLE_CONSOLE
         ble.waitForEvent();
#endif // #if NEED_BLE_CONSOLE
    }
}

