// BLE Condensation Monitor
// https://developer.mbed.org/users/takafuminaka/code/BLE_Condensation_Monitor/
// T.Naka

#include "mbed.h"
#include "HDC1000.h"
#include "BME280.h"
#include "AQM0802.h"

#include "VaporCondition.h"

#define NEED_CONSOLE_OUTPUT 1
#define NEED_BLE_CONSOLE 1
#define NEED_LCD_OUTPUT 1

#define TYPE_HDC1000 1
#define TYPE_BME280 2

// #define INSIDE_SENSOR TYPE_HDC1000
#define INSIDE_SENSOR TYPE_BME280

#if NEED_CONSOLE_OUTPUT
Serial  pc(USBTX, USBRX);
#define PC(...) { pc.printf(__VA_ARGS__); }
#else
#define PC(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

#if NEED_BLE_CONSOLE
#include "BLEDevice.h"
#include "UARTService.h"
#define BLEC(...) { char __blecstr[32]; sprintf(__blecstr,__VA_ARGS__); if (uart) uart->write(__blecstr, strlen(__blecstr)); }
#else
#define BLEC(...) /* nothing */
#endif /* #if NEED_BLE_CONSOLE */

#if defined(TARGET_LPC1768)
I2C i2c(p28, p27);
// BME280 sensor(p28, p27, 0x76 << 1);
#else
I2C i2c(I2C_SDA0, I2C_SCL0);
// BME280 sensor(I2C_SDA0, I2C_SCL0, 0x76 << 1);
#endif

BME280 sensor_out(I2C_SDA0, I2C_SCL0, 0x76 << 1);

#if INSIDE_SENSOR == TYPE_HDC1000
HDC1000 sensor_in(i2c);
#elif INSIDE_SENSOR == TYPE_BME280
BME280 sensor_in(I2C_SDA0, I2C_SCL0, 0x77 << 1);
#endif

#if NEED_LCD_OUTPUT
AQM0802 lcd(i2c);
#endif

DigitalOut led1(LED1);
DigitalOut led2(LED2);

#if NEED_BLE_CONSOLE
BLEDevice  ble;
UARTService *uart;

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    PC("Disconnected!\n\r");
    PC("Restarting the advertising process\n\r");
    ble.startAdvertising();
}

#endif /* #if NEED_BLE_CONSOLE */

int main()
{
    float Tdp_o =0.;
    float Tdp_i =0.;
    int cautions = 0;
    int warnings = 0;
    int warn_wid = 20;
    char msg1[10],msg2[10];
    int mode=0;
    int skip=0;
    char msg[4][2][10];
    char bmsg[4][20];
    int skipf[4];

    float Tcur;
    float Tdp;
    char *sTcur;
    char *sWin;
    char *ssTcur;
    char *ssWin;
    char *bsTcur;
    char *bsWin;

    int skipped = 0;

    VaporCondition Inside;
    VaporCondition Outside;

    // LED Check
    led1 = 1;
    led2 = 1;
    wait(3);

    led1 = 0;
    led2 = 0;

    i2c.frequency(100000);

#if NEED_BLE_CONSOLE
    // Setup BLE //
    ble.init();
    ble.onDisconnection(disconnectionCallback);

    uart = new UARTService(ble);

    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)"BLE Condensation Monitor", sizeof("BLE Condensation Monitor") - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));

    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();
#endif /* #if NEED_BLE_CONSOLE */

    while(1) {
        if ( skipped == 0 ) {
            // Get data

            Outside.t = sensor_out.getTemperature();
            Outside.h = sensor_out.getHumidity();

            Outside.p = sensor_out.getPressure();

#if INSIDE_SENSOR == TYPE_HDC1000
            Inside.p = sensor_out.getPressure(); // Usually Pressures are same between inside and outside.
            Inside.t = float(sensor_in.temperature())/0x10000*165-40;
            Inside.h = float(sensor_in.humidity())/0x10000*100;
            PC("In: %2.2f degC, %2.2f %%\r\n", Inside.t, Inside.h);
#elif INSIDE_SENSOR == TYPE_BME280
            Inside.t = sensor_in.getTemperature();
            Inside.h = sensor_in.getHumidity();
            Inside.p = sensor_in.getPressure();
            PC("In: %2.2f degC, %2.2f %%, %04.2f hPa\r\n", Inside.t, Inside.h, Inside.p);
#endif

            PC("Out: %2.2f degC, %2.2f %%, %04.2f hPa\r\n", Outside.t, Outside.h, Outside.p);

            PC("Humidity Ratio [g/kg] : In %2.2f Out %2.2f \r\n", Inside.Rh(), Outside.Rh());
            Tdp_o = Outside.Tdp();
            Tdp_i = Inside.Tdp();
            PC("Due Point Temperature [degC] : In %2.2f Out %2.2f \r\n", Tdp_i, Tdp_o);

            // print catuions and warnings //
            cautions = 0;
            warnings = 0;

            for(int ii=0; ii<4; ii++) {
                if ( (ii % 2) == 1 ) {
                    Tcur = Outside.t;   // 1 and 3
                    sTcur = "Outside";
                    ssTcur = "Out";
                    bsTcur = "Out";
                } else {
                    Tcur = Inside.t;    // 0 and 2
                    sTcur = "Inside";
                    ssTcur = "In";
                    bsTcur = "In";
                }

                if ( ii / 2 ) {
                    sWin = " Window"; // 2 and 3
                    ssWin = "@Win";
                    bsWin = "Wind.";
                } else {
                    sWin = "";         // 0 and 1
                    ssWin = "";
                    bsWin = "";
                }

                if ( ii / 2 == ii %2 ) {
                    Tdp = Tdp_i;    // 0 and 3
                } else {
                    Tdp = Tdp_o;    // 1 and 1
                }

                if ( Tdp >= Tcur - warn_wid ) {
                    skipf[ii] = 0;
                    if ( Tdp >= Tcur ) {
                        PC("Condensation at %s %s\r\n",sTcur,sWin);
                        sprintf(msg[ii][0],"Condns!!");
                        sprintf(msg[ii][1],"%s%s",ssTcur,ssWin);
                        sprintf(bmsg[ii],"Cond. at %s%s",bsTcur,bsWin);
                        cautions ++;
                    } else {
                        PC("%4.1f degC to Condensation at %s%s\r\n", Tcur - Tdp, sTcur, sWin);
                        sprintf(msg[ii][0],"Cto%4.1fC",Tcur-Tdp);
                        sprintf(msg[ii][1],"%s%s\0",ssTcur,ssWin);
                        sprintf(bmsg[ii],"%4.1ftoC.at%s%s",Tcur - Tdp, bsTcur, bsWin);
                        warnings ++;
                    }
                } else {
                    skipf[ii] = 1;
                }
            }

            PC("\r\n");

            if ( cautions > 0 ) {
                led2 = 1;
            } else {
                led2 = 0;
            }

            if ( warnings > 0 ) {
                led1 = 1;
            } else {
                led1 = 0;
            }

        }
        // LCD print
        switch(mode) {
            case (0):
                skip = 1;
                break;

            case (1):
                sprintf(msg1,"Ti %4.1fC",Inside.t);
                sprintf(msg2,"To %4.1fC",Outside.t);
                BLEC("Temp.In%4.1f Out%4.1f\n",Inside.t, Outside.t);
                break;

            case (2):
                sprintf(msg1,"Hi %4.1f%%",Inside.h);
                sprintf(msg2,"Ho %4.1f%%",Outside.h);
                BLEC("Hum.In%2.0f%% Out%2.0f%%\n",Inside.h, Outside.h);
                break;

            case (3):
                sprintf(msg1,"Po%6.1f",Outside.p);
                sprintf(msg2,"   [hPa]");
                BLEC("Prs.Out%6.1fhPa\n",Outside.p);
                break;

            case (4):
#if INSIDE_SENSOR == TYPE_BME280 
                sprintf(msg1,"Pi%6.1f",Inside.p);
                sprintf(msg2,"   [hPa]");
                BLEC("Prs.In %6.1fhPa\n",Inside.p);
#elif INSIDE_SENSOR == TYPE_HDC1000
                skip = 1;
#endif
                break;

            case (5):
                sprintf(msg1,"Dpi%4.1fC",Tdp_i);
                sprintf(msg2,"Dpo%4.1fC",Tdp_o);
                BLEC("DP. In%4.1f Out%4.1f\n",Tdp_i, Tdp_o);
                break;

            case (6):
            case (7):
            case (8):
            case (9):
                int ii = mode - 6;
                skip = skipf[ii];
                sprintf(msg1,"%8s",msg[ii][0]);
                sprintf(msg2,"%8s",msg[ii][1]);
                BLEC("%s\n",bmsg[ii]);
                break;
        }
        mode++;
        if ( mode > 9 ) {
            mode = 0;
        }

        if ( skip == 0 ) {
#if NEED_LCD_OUTPUT
            lcd.locate(0,0);
            lcd.print(msg1);
            lcd.locate(0,1);
            lcd.print(msg2);

#endif
            skipped = 0;
            wait(3);
        } else {
            skip = 0;
            skipped = 1;
        }
    }

}