#include "mbed.h"
#include "TouchButton.h"

// Example of a FSM mimicing the traffic light.
// Touch the slider for requesting to walk through
/*
The purpose is to reproduce the behaviour of a traffic light both
on the cars and the pedestrians sides. When placing the board with
the slider on the bottom, it the left side of the slider the LED
would reproduce the pedestrians side, turning the LED into green
first, then blinking it and finally turning it into red. If the
right side is touched the cars side is reproduced, turning the LED
yellow, then red, and finally green. Time is given for cars to go
through for some time although the traffic light has become green
for cars and you touch the slider close to that moment -- some time is
needed for the changes to take on.
At first, on the cars side, is nothing is touched the traffic lights
will be allowing cars to go through indefinitely.
*/

// Working with the FRDM-KL25Z board...

#define PEDESTRIANS 0
#define CARS        1

unsigned char lastpressed, pedestrianrequest, whichside = 0;
TouchButton slider;
PwmOut r (LED_RED);
PwmOut g (LED_GREEN);
PwmOut b (LED_BLUE);

unsigned char cnt;

enum TL_States_t {Init, DontWalk, WaitBut, WaitSafe, Walk, BlinkDontWalk } tl_state;

inline void SetLedRed() {
    r=0;
    g=b=1;
}

inline void SetLedGreen() {
    g=0;
    r=b=1;
}

inline void SetLedYellow() {
    r=g=0;
    b=1;
}

inline void SwitchOffLed() {
    g=r=b=1;
}

int TickFunc_TrafficLights()
{
    // Next-state setting.
    switch (tl_state) {
        case Init:
          tl_state = DontWalk;
          cnt = 0;
          // Setting the LED red
          r=0;
          g=b=1;
          break;
        case DontWalk:
          if (lastpressed) {
            pedestrianrequest = 1;
            whichside = lastpressed - 1;
          };
          if (cnt < 20) {
            tl_state = DontWalk;
            cnt++;
          }
          else {
            if (!pedestrianrequest)
              tl_state = WaitBut;
            else
              tl_state = WaitSafe;
            cnt = 0;
          };
          if (whichside == CARS)
            SetLedGreen();
          else
            SetLedRed();
         break;
        case WaitBut:
          if (!lastpressed)
            tl_state = WaitBut;
          else {
            whichside = lastpressed - 1;
            // We'll use the whichside value 0 for showing the pedestrian side
            // and the whichside value 1 for the cars side.
            tl_state = WaitSafe;
            // cnt = 0; // Already done.
          };
          break;
        case WaitSafe:
          // We start attending the pedestrians request here...
          pedestrianrequest = 0;
          if (cnt < 3) {
            tl_state = WaitSafe;
            cnt++;
          }
          else {
            tl_state = Walk;
          };
          if (whichside == CARS)
            SetLedYellow();
          else
            SetLedRed();
         break;
        case Walk:
          if (cnt < 20) {
            tl_state = Walk;
            cnt++;
          }
          else {
            tl_state = BlinkDontWalk;
            cnt = 0;
          };
          if (whichside == CARS)
            SetLedRed();
          else
            SetLedGreen();
         break;
        case BlinkDontWalk:
          if (cnt < 8) {
            tl_state = BlinkDontWalk;
            cnt++;
          }
          else {
            tl_state = DontWalk;
            cnt = 0;
          }
          if (whichside == PEDESTRIANS) {
            r=b=1;
            g = g ? 0 : 1;
          }
          break;
        default:
          break;
    }
    return 0;
}

int main() {
    tl_state = Init;
    r=g=b=1;
    while(1) {
        lastpressed = slider.PressedButton() < 0.5 ? 0 : 1;
        lastpressed = slider.PressedButton() < 2 ? lastpressed : 2;
        // lastpressed will be 0 in case that the slider is not touched, 1 in case
        // the finger is on the left, and 2 if it is on the right.
        TickFunc_TrafficLights();
        wait(0.5);
    }
}
