#ifndef LEG_PACKETS_H
#define LEG_PACKETS_H


/**
Constants and data structures shared between the local control system and the central one (on the mbed).
This include file is also loaded by the mbed.

*/

// Magic number used to confirm that the mbed has received a correct packet.  Not clear why we need this checking, though...
#define I2C_MAGIC_NUMBER 0xA5

/**
Addresses for the different legs
*/
#define I2C_ADDR_EAST 0x45
#define I2C_ADDR_WEST 0x57
#define I2C_ADDR_SOUTH 0x53

/**
Types of the packets that are sent from the central controller to the local controller
*/

namespace nsPacketsLeg {
typedef enum {SET_GOAL=0, SET_LIFT_PARAMS, GET_LIFT_PARAMS, SET_WHEEL_PARAMS, GET_WHEEL_PARAMS, QUERY_STATE, SET_GOAL_QUERY_STATE, ESTOP_ON, ESTOP_OFF, NPACKET_TYPES} LEG_PACKET_TYPE;
}

/**
State of some actuator
*/
typedef struct{
  int32_t pos;
  int32_t vel;
}ActuatorState_t;

/**
State of the entire leg.

Sent in response to a QUERY_STATE or a SET_GOAL_QUERY_STATE packet
*/
typedef struct{
  ActuatorState_t lift;
  ActuatorState_t wheel;
  bool cliff;
  bool limit;
  bool estop;         // Leg reports current state
  uint8_t magic;
}LegState_t;

/**
Goal of the leg
*/
typedef struct{
  int32_t liftPos;
  int32_t wheelVel;
}Goal_t;

/**
Leg control parameters
*/
typedef struct{
  int16_t Kp;
  int16_t Kv;
  int16_t Ki;
  int16_t deadband;
  int16_t min_val;     // Minimum control signal
  int16_t max_val;     // Maximum control signal
  int16_t max_error;   // Maximum position error used in the "P" component of the controller
  int16_t max_error_accum;  // Maximum error integral
  int16_t max_accel;   // Maximum acceleration (used for wheel)
  int16_t min_goal;    // Minimum allowd goal value
  int16_t max_goal;     // Maximum allowed goal value
}LegControlParams_t;

/**
Communication packet from central controller (mbed) to the local Teensy controller.

Note: when a packet is transmitted, only the necessary bytes are transferred for a given type.
*/

typedef struct{
  nsPacketsLeg::LEG_PACKET_TYPE type;
  union {
    Goal_t as_goal;                          // SET_GOAL, SET_GOAL_QUERY_STATE
    LegControlParams_t as_leg_params;        // SET_LEG_PARAMS
  }contents;
}LegPacket_t;

#endif

