/*******************************************************************************
 * Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef _MAX32625PICO_H_
#define _MAX32625PICO_H_

#include "mbed.h"

/**
 * @brief MAX32625PICO Board Support Library
 *
 * @details The MAX32625PICO is a rapid development application board for
 * ultra low power wearable applications.  It includes common peripherals and
 * expansion connectors all power optimized for getting the longest life from
 * the battery.  This library configures the power and I/O for the board.
 * <br>https://www.maximintegrated.com/MAX32625PICO
 *
 * @code
 * #include "mbed.h"
 * #include "max32625pico.h"
 *
 * DigitalOut led1(LED1);
 * MAX32625PICO pico(MAX32625PICO::IOH_SWD_IN, MAX32625PICO::VIO_1V8, MAX32625PICO::VIO_IOH);
 *
 * // main() runs in its own thread in the OS
 * // (note the calls to Thread::wait below for delays)
 * int main()
 * {
 *     // initialize power and I/O on MAX32625PICO board
 *     pico.init();
 *
 *     while (true) {
 *         led1 = !led1;
 *         Thread::wait(500);
 *     }
 * }
 * @endcode
 */
class MAX32625PICO
{
public:
// MAX32625PICO configuration utilities

    /**
      * @brief   IO Voltage
      * @details Enumerated options for operating voltage
      */
    typedef enum {
        VIO_1V8 = 0x00,    ///< 1.8V IO (local)
        VIO_IOH = 0x01,    ///< Use VDDIOH (from DIP pin 1, or SWD pin1, or local 3.3V)
    } vio_t;

    /**
      * @brief   VDDIOH Mode
      * @details Enumerated options for VDDIOH configuration mode
      */
    typedef enum {
        IOH_OFF = 0x00,     ///< No connections to VDDIOH
        IOH_DIP_IN = 0x02,  ///< VDDIOH input from DIP pin 1
        IOH_SWD_IN = 0x03,  ///< VDDIOH input from SWD pin 1
        IOH_3V3 = 0x04,     ///< VDDIOH = 3.3V from local supply
        IOH_DIP_OUT = 0x06, ///< VDDIOH = 3.3V output to DIP pin 1
        IOH_SWD_OUT = 0x07, ///< VDDIOH = 3.3V output to SWD pin 1
    } vddioh_mode_t;

    /**
        * MAX32625PICO constructor.
        *
        */
    MAX32625PICO();

    /**
        * MAX32625PICO constructor.
        *
        */
    MAX32625PICO(vddioh_mode_t iohMode, vio_t dipVio, vio_t swdVio);

    /**
        * MAX32625PICO destructor.
        */
    ~MAX32625PICO();

    /**
     * @brief   Initialize MAX32625PICO board
     * @details Initializes PMIC and I/O on MAX32625PICO board.
     *  Configures PMIC to enable LDO2 and LDO3 at 3.3V.
     *  Disables resisitive pulldown on MON(AIN_0)
     *  Sets default I/O voltages to 3V3 for micro SD card.
     *  Sets I/O voltage for header pins to hdrVio specified.
     * @param iohMode Configuration mode for VDDIOH
     * @param dipVio I/O voltage for DIP pins
     * @param swdVio I/O voltage for SWD pins
     * @returns 0 if no errors, -1 if error.
    */
    int init(vddioh_mode_t iohMode, vio_t dipVio, vio_t swdVio);

    /**
     * @brief   Sets I/O Voltage
     * @details Sets the voltage rail to be used for a given pin.
     *  VIO_1V8 selects VDDIO which is supplied 1.8V locally,
     *  VIO_IOH selects VDDIOH which is supplied externally or from the 3.3V supply/
     * @param   pin Pin whose voltage supply is being assigned.
     * @param   vio Voltage rail to be used for specified pin.
     * @returns 0 if no errors, -1 if error.
    */
    int vddioh(PinName pin, vio_t vio);

    /// Digital output controlling local 3.3V IOH supply
    DigitalOut en3V3;
    /// Digital output controlling external IOH supply
    DigitalOut enIOH;
    /// Digital output selecting external IOH supply
    DigitalOut selSWD;

};

#endif /* _MAX32625PICO_H_ */
