#include "mbed.h"
#include "max32630fthr.h"
#include "SDFileSystem.h"
#include "USBKeyboard.h"

MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);

// Hardware serial port over DAPLink
Serial daplink(P2_1, P2_0);

DigitalOut rLED(LED1, LED_ON);
DigitalOut gLED(LED2, LED_OFF);
DigitalOut bLED(LED3, LED_OFF);
DigitalIn button(SW1, PullUp);
AnalogIn monIn(AIN_0);
AnalogIn vddbIn(AIN_6);
AnalogIn vdd18In(AIN_7);
AnalogIn vdd12In(AIN_8);
AnalogIn vrtcIn(AIN_9);

SDFileSystem sd(P0_5, P0_6, P0_4, P0_7, "sd");  // mosi, miso, sclk, cs
USBKeyboard keyboard;

#define BMI160_I2C_ADDR 0xD0

// Function to report failure
void testFailed()
{
    daplink.printf("\r\n! Test Failed !\r\n");
    rLED = LED_ON;
    gLED = LED_OFF;
    bLED = LED_OFF;
    while(1) {
        wait_ms(500);
        gLED = !gLED;
    }
}

// Function to read monitor port
float readMon(MAX14690::monCfg_t monCfg)
{
    pegasus.max14690.monSet(monCfg, MAX14690::MON_DIV4);
    wait_ms(5);
    return (4.8f * monIn);
}

// main() runs in its own thread in the OS
// (note the calls to Thread::wait below for delays)
int main()
{
    float aIn;
    char dataBuf[2];
    int ledCnt = 0;
    
    daplink.printf("Initializing MAX32630FTHR\r\n");
//    pegasus.init(MAX32630FTHR::VIO_3V3);
    daplink.printf("Checking Supplies\r\n");
    aIn = 2.4f * vdd12In;
    daplink.printf("vdd12 = %f\r\n", aIn);
    if ((aIn < 1.0f) || (aIn > 1.4f)) {
        testFailed();
    }
    aIn = 2.4f * vdd18In;
    daplink.printf("vdd18 = %f\r\n", aIn);
    if ((aIn < 1.6f) || (aIn > 2.0f)) {
        testFailed();
    }
    aIn = 4.8f * vddbIn;
    daplink.printf("vddb = %f\r\n", aIn);
    if ((aIn < 3.0f) || (aIn > 3.6f)) {
        testFailed();
    }
    aIn = 2.4f * vrtcIn;
    daplink.printf("vrtc = %f\r\n", aIn);
    if ((aIn < 1.6f) || (aIn > 2.0f)) {
        testFailed();
    }
    aIn = readMon(MAX14690::MON_SYS);
    daplink.printf("sys = %f\r\n", aIn);
    if ((aIn < 3.0f) || (aIn > 5.0f)) {
        testFailed();
    }
    aIn = readMon(MAX14690::MON_BUCK1);
    daplink.printf("buck1 = %f\r\n", aIn);
    if ((aIn < 1.0f) || (aIn > 1.4f)) {
        testFailed();
    }
    aIn = readMon(MAX14690::MON_BUCK2);
    daplink.printf("buck2 = %f\r\n", aIn);
    if ((aIn < 1.6f) || (aIn > 2.0f)) {
        testFailed();
    }
    aIn = readMon(MAX14690::MON_LDO1);
    daplink.printf("ldo1 = %f\r\n", aIn);
    if ((aIn < 1.6f) || (aIn > 2.0f)) {
        testFailed();
    }
    aIn = readMon(MAX14690::MON_LDO2);
    daplink.printf("ldo2 = %f\r\n", aIn);
    if ((aIn < 3.0f) || (aIn > 3.6f)) {
        testFailed();
    }
    aIn = readMon(MAX14690::MON_LDO3);
    daplink.printf("ldo3 = %f\r\n", aIn);
    aIn = readMon(MAX14690::MON_LDO3);

    bLED = LED_ON;

    daplink.printf("Checking micro SD Card\r\n");
    FILE *fp = fopen("/sd/myfile.txt", "r");
    if(fp == NULL) {
        daplink.printf("Unable to open 'myfile.txt'\r\n");
        testFailed();
    }
    daplink.printf("micro SD Card present\r\n");
    rLED = LED_OFF;

    daplink.printf("Checking BMI160\r\n");
    dataBuf[0] = 0x00;
    if (pegasus.i2c.write(BMI160_I2C_ADDR, dataBuf, 1, 1) != 0) testFailed();
    if (pegasus.i2c.read(BMI160_I2C_ADDR, dataBuf, 1) != 0) testFailed();
    if (dataBuf[0] != 0xD1) testFailed();
    daplink.printf("BMI160 present\r\n");
    gLED = LED_ON;

    

    daplink.printf("Self Test Passed\r\n");
    rLED = LED_ON;
    gLED = LED_ON;
    bLED = LED_ON;

    while (true) {
        if (!button) {
            keyboard.mediaControl(KEY_MUTE);
            ledCnt++;
            gLED = (ledCnt & 1);
            bLED = (ledCnt & 2);
            rLED = gLED ^ bLED;
            wait_ms(500);
        } else {
            wait_ms(50);
        }
    }
}

