//============================================
// eightDotMatrixLed Library Example
//
// This program is used to control the LED 32.
//
//
// <schematic>
//                               --|>|-- 
//                                A   K
//   seg A(p5) -----R(200[ohm])--- LED ----- com0(p13)
//               |
//               -- R(200[ohm])--- LED ----- com1(p14)
//               |
//               -- R(200[ohm])--- LED ----- com2(p28)
//               |
//               -- R(200[ohm])--- LED ----- com3(p27)
//
//
//   same : segB(p6), segC(p7), segD(p8), segE(p9), segF(p10), segG(p11), segH(p12)
//
// <Description of LED control>
// com0 Led off to on ,on to off smooth (lighting = 0 to 100 [%]
// com1 Led off to on smooth,   on to off hard
// com2 Led off to on, on to off smooth (lighting = 0 to gray data)
// com3 gray data movement
//
//=============================================

#include "mbed.h"

#include "EightDotMatrixLed.h"

//                           common type (0:anode common 1:cathode common)
//                           |  
//                           |   segA segB segC segD segE segF segG segh com1 com2 com3 com4  (com5 to com8 = disable)                          
//                           |   |    |    |    |    |    |    |    |    |    |    |    | 
EightDotMatrixLed segmentled(1, p5,  p6,  p7,  p8,  p9, p10, p11, p12, p13, p14, p28, p27);

// com0 Led off to on ,on to off smooth (lighting = 0 to 100 [%]
// com1 Led off to on smooth,   on to off hard
// com2 Led off to on, on to off smooth (lighting = 0 to gray data)
// com3 gray data movement


uint8_t D_dotGrayData[4][8] =   {
                                    {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},   // com0 disable
                                    {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},   // com1 disable
                                    {0,      14,   29,   43,   57,   71,   86,  100},   // com2 enable (0 to 100 [%])
                                    {0,      14,   29,   43,   57,   71,   86,  100},   // com3 enable (0 to 100 [%])
                                 };
uint8_t D_dotDigitalData[4][8] = {
                                    {0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // com0 0:off, 1:on
                                    {0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},   // com1 0:disalbe 1:on                                    
                                    {0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},   // com2 0:off, 1:on
                                    {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}    // com3 disable
                                 };

Timer timer;    // data change timer
    
int main() {
    uint8_t seg;
    uint8_t wk0, wk1, wk2, wk3;
    
    timer.start();

    while(1) {
        // After 500[ms] to start the process
        if(timer.read_ms() >= 500){
            timer.reset();

            // com0 Led off to on ,on to off smooth (lighting = 0 to 100 [%]
            wk0 = D_dotDigitalData[0][7];
            for(seg = 7; seg > 0; seg--){
                D_dotDigitalData[0][seg] = D_dotDigitalData[0][seg - 1];
            }
            D_dotDigitalData[0][0] = wk0;
            
            // com1 Led off to on smooth,   on to off hard
            wk1 = D_dotDigitalData[1][7];
            for(seg = 7; seg > 0; seg--){
                D_dotDigitalData[1][seg] = D_dotDigitalData[1][seg - 1];
            }
            D_dotDigitalData[1][0] = wk1;
            
            // com2 Led off to on, on to off smooth (lighting = 0 to gray data)
            wk2 = D_dotDigitalData[2][7];
            for(seg = 7; seg > 0; seg--){
                D_dotDigitalData[2][seg] = D_dotDigitalData[2][seg - 1];
            }
            D_dotDigitalData[2][0] = wk2;

            // com3 gray data movement
            wk3 = D_dotGrayData[3][7];
            for(seg = 7; seg > 0; seg--){
                D_dotGrayData[3][seg] = D_dotGrayData[3][seg - 1];
            }
            D_dotGrayData[3][0] = wk3;
        }
        
        // This function, please repeat the process in less than 1ms.
        segmentled.EightDotMatrixLed_main((uint8_t*)D_dotGrayData, (uint8_t*)D_dotDigitalData);      
 
    }
}
