/* SwAnalog Library
 * Copyright (c) 2012 suupen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/***********************************************************************/
/*                                                                     */
/*    SwAnalog.h                                                       */
/*                                                                     */
/*                                                                     */
/*    2012/02/12 : V1.0                                                */
/***********************************************************************/
#ifndef _SWANALOG_H
#define _SWANALOG_H

#include "mbed.h"

//#define LPC1768

/** SWANALOG control class, based on a "mbed function"
*
* Example:
* @code
* // *******************************************************************
* // SwAnalogInput Library example program
* //  mbed no analog port de 3hon no sw wo ninsiki suru.
* //  6hon no analog port (p15 - p20) de 6 * 3 = 18 ko no sw ninsiki ga dekiru.
* //
* // <schematic>
* //   -.- mbed VOUT(+3.3[V])
* //    |                                               |--------------------> mbed p20(ADinput)
* //    |   ---------       ---------       ---------   |   ---------
* //    .---| Rsw2  |---.---| Rsw1  |---.---| Rsw0  |---.---| Rout  |----|
* //    |   ---------   |   ---------   |   ---------   |   ---------    |
* //    |     ----      |     -----     |     -----     |                |
* //    |-----o  o------.-----o  o------.-----o  o------|              -----
* //           SW2            SW1              SW0                      mbed GND(0[V])
* // 
* //  Rsw2 : 8.2[kohm], Rsw1 = 3.9[kohm], Rsw0 = 2.0[kohm], Rout = 1.0[kohm] (R no seido ha +-1[%])
* //
* // <program no naiyo>
* //  mbed LED1 : sw0 no level de tento  sw Off = led Off, sw On = led On
* //  mbed LED2 : sw1 no level de tento  sw Off = led Off, sw On = led On
* //  mbed LED3 : sw2 no level de tento  sw Off = led Off, sw On = led On
* //  mbed LED4 : sw0,sw1,sw2 no Off kara On henka de led tento jotai wo hanten.
* //
* // 
* // <rireki>
* // 2012/02/12 : syohan
* // *********************************************************************
*
* #include "mbed.h"
* #include "SwAnalog.h"
*
* DigitalOut led1(LED1);
* DigitalOut led2(LED2);
* DigitalOut led3(LED3);
* DigitalOut led4(LED4);
*
* SwAnalog sw(p20);  // p20(adinput) :sw0,sw1,sw2
*                   
* int main() {
*     while(1) {
*         // sw level and edge data refresh
*         sw.refreshEdgeData();
*         
*         // tact action (sw0 level = on : led1 = on)
*         led1 = sw.checkLevel(0);
*         
*         // tact action (sw1 level = off : led2 = on)
*         led2 = sw.checkLevel(1);
*         
*         // tact action (sw2 level = off : led3 = on)
*         led3 = sw.checkLevel(2);
*         
*         // toggle action (sw0 to sw3 level Off to On : led4 invert )
*         if((sw.checkEdgeOn(0) == 1)   // sw0
*         || (sw.checkEdgeOn(1) == 1)   // sw1
*         || (sw.checkEdgeOn(2) == 1)   // sw2
*         ){
*              led4 = !led4;
*         }
*     }
* }
* @endcode
*/

class SwAnalog {
public:

    /** Create a sw analog input object connected to the specified analog Input pin
    *
    * @param PinName adinput0 : analog input pin(pin15 to pin20) : sw0 kara sw2   no ninsiki 
    * @param PinName adinput1 : analog input pin(pin15 to pin20) : sw3 kara sw5   no ninsiki 
    * @param PinName adinput2 : analog input pin(pin15 to pin20) : sw6 kara sw8   no ninsiki
    * @param PinName adinput3 : analog input pin(pin15 to pin20) : sw9 kara sw11  no ninsiki     
    * @param PinName adinput4 : analog input pin(pin15 to pin20) : sw12 kara sw14 no ninsiki     
    * @param PinName adinput5 : analog input pin(pin15 to pin20) : sw15 kara sw17 no ninsiki     
    *
    * Recognition of the SW period is 10ms
    */
  
#ifdef LPC1768
    SwAnalog(PinName adinput0  = NC, PinName adinput1  = NC, PinName adinput2 = NC, PinName adinput3  = NC, PinName adinput4  = NC,
              PinName adinput5  = NC
              );
#else // ~LPC1768
   SwAnalog(PinName adinput0  = NC, PinName adinput1  = NC
              );
#endif // LPC1768
    /** refresh edge data
    *
    * @param none 
    * @param return none
    *
    *  main de edge data wo tukau maeni jiko suru
    */
    void refreshEdgeData(void);
    
    /** Check Off to On edge
    *
    * @param uint8_t swNo     : 0:sw0, 1:sw1, ... ,17:sw17     
    * @param return uint8_t  On edge check  0: edge Nasi  1: edge Ari
    *
    */
    uint8_t checkEdgeOn(uint8_t swNo);

    /** Check On to Off edge
    *
    * @param uint8_t swNo     : 0:sw0, 1:sw1, ... ,17:sw17   
    * @param return uint8_t Off edge check   0 : Nasi   1 : Ari
    *
    */
    uint8_t checkEdgeOff(uint8_t swNo);
    
    /** Check sw Level
    *
    * @param uint8_t swNo     : 0:sw0, 1:sw1, ... ,17:sw17   
    * @param return uint8_t sw level check   0 : Off   1 : On
    *
    */    
    uint8_t checkLevel(uint8_t swNo);
    
    /** SW Number call name
    *
    * @param adinput0 : Z_sw1 to Z_sw3
    * @param adinput1 : Z_sw4 to Z_sw6
    * @param  ....
    * @param adinput5 : Z_sw16 tp Z_sw18
    */
#ifdef LPC1768
    enum{
        Z_sw0,  // adinput1 no sw
        Z_sw1,
        Z_sw2,
        
        Z_sw3,  // adinput2 no sw
        Z_sw4,
        Z_sw5,
        
        Z_sw6,  // adinput3 no sw
        Z_sw7,
        Z_sw8,
        
        Z_sw9, // adinput4 no sw
        Z_sw10,
        Z_sw11,
        
        Z_sw12, // adinput5 no sw
        Z_sw13,
        Z_sw14,
        
        Z_sw15, // adinput6 no sw
        Z_sw16,
        Z_sw17
        };
#else // ~LPC1768
    enum{
        Z_sw0,  // adinput1 no sw
        Z_sw1,
        Z_sw2,
        
        Z_sw3,  // adinput2 no sw
        Z_sw4,
        Z_sw5
    };
#endif // LPC1768
//protected:    

#ifdef LPC1768
   AnalogIn _adinput0;
   AnalogIn _adinput1;
   AnalogIn _adinput2;
   AnalogIn _adinput3;
   AnalogIn _adinput4;

   AnalogIn _adinput5;
#else // ~LPC1768
   AnalogIn _adinput0;
   AnalogIn _adinput1;
#endif   
   
   Ticker swCheckTimer;
   
   void input(void);

   #define Z_matchcycle (10000) // 10000[us](10[ms]) to 100000[us](100[ms])  1[us]/count
   
   uint8_t D_swPinSuu;         // touroku sareta Sw Pin Suu  1 to Z_swPinSuuMax
   #define Z_swPinSuuMax  (6)  // SW warituke pin suu max
   #define Z_swInNoMax    (3)  // 1pin atari no sw setuzoku suu (1pin ni 3ko no sw setuzoku)
   
   uint8_t B_kariLevel[Z_swPinSuuMax * Z_swInNoMax]; // kakutei mae no ninsiki Level 0bit:saisin(t) 1bit:t-1, ... ,7bit:t-7  0:Off  1:On
    // match number define
    //#define Z_itchiPattern     (0x03)   // 2kai itch
    #define Z_itchiPattern    (0x07)  // 3kai itchi
    //#define Z_itchiPattern    (0x0f)  // 4kai itchi
    //#define Z_itchiPattern    (0x1f)  // 5kai itchi
    //#define Z_itchiPattern    (0x3f)  // 6kai itchi
    //#define Z_itchiPattern    (0x7f)  // 7kai itchi
    //#define Z_itchiPattern    (0xff)  // 8kai itchi
   
   // sw level data
   uint8_t D_nowLevel[Z_swPinSuuMax * Z_swInNoMax];  // saisin no kakutei Level 0:Off  1:On
   uint8_t D_oldLevel[Z_swPinSuuMax * Z_swInNoMax];  // zenkai no kakutei Level 0:Off  1:On
    #define Z_levelOff (0)
    #define Z_levelOn  (1)
   
   // sw edge data
   // swDigital.c naibu hensu 
   uint8_t B_edgeOn[Z_swPinSuuMax * Z_swInNoMax];    // off kara on  no ninsiki(on edge)  0:Nasi  1:Ari
   uint8_t B_edgeOff[Z_swPinSuuMax * Z_swInNoMax];   // on  kara off no ninsiki(off edge) 0:Nasi  1:Ari
   // user use hensu
   uint8_t D_edgeOn[Z_swPinSuuMax * Z_swInNoMax];    // off kara on  no ninsiki(on edge)  0:Nasi  1:Ari
   uint8_t D_edgeOff[Z_swPinSuuMax * Z_swInNoMax];   // on  kara off no ninsiki(off edge) 0:Nasi  1:Ari
    #define Z_edgeNasi (0)
    #define Z_edgeAri  (1)

    //------------------
    // Resistor network
    //------------------
    
    //   -.- mbed VOUT(+3.3[V])
    //    |                                               |--------------------> mbed p15 - p20(analog port)
    //    |   ---------       ---------       ---------   |   ---------
    //    .---| Rsw2  |---.---| Rsw1  |---.---| Rsw0  |---.---| Rout  |----|
    //    |   ---------   |   ---------   |   ---------   |   ---------    |
    //    |     ----      |     -----     |     -----     |                |
    //    |-----o  o------.-----o  o------.-----o  o------|              -----
    //           SW2            SW1              SW0                      mbed GND(0[V])
    // 
    //    |                                                                |
    //    |<----------------------- Rall --------------------------------->|    
    //    |                          |                                     |
    //                               ----> Z_R0 to Z_R7
    
    
    #define Z_Rsw2 (8200.0F)   // SW2 no R (1/1 [ohm]/count)
    #define Z_Rsw1 (3900.0F)   // SW1 no R (1/1 [ohm]/count)
    #define Z_Rsw0 (2000.0F)   // adinput1 no R (1/1 [ohm]/count)
    #define Z_Rout (1000.0F)   // Vout no R (1/1 [ohm]/count)
    //Z_Rsw2,Z_Rsw1,Z_Rsw0,Z_Rout niwa +-1[%]no seido no teiko wo tukau koto
    
    #define Z_gosaMax (1.020F) // Z_Rout(max) / Z_Rx(min) = (Z_Rout * 1.01) / (Z_Rx * 0.99) = (Z_Rout / Z_Rx) * 1.020
    #define Z_gosaMin (0.990F) // Z_Rout(min) / Z_Rx(max) = (Z_Rout * 0.99) / (Z_Rx * 1.01) = (Z_Rout / Z_Rx) * 0.980
    
    // Rall keisanchi
                                                           // SW2  SW1  SW0
    #define Z_R0 ((Z_Rsw2 + Z_Rsw1 + Z_Rsw0 + Z_Rout))     // OFF  OFF  OFF
    #define Z_R1 ((Z_Rsw2 + Z_Rsw1 + 0      + Z_Rout))     // OFF  OFF   ON
    #define Z_R2 ((Z_Rsw2 + 0      + Z_Rsw0 + Z_Rout))     // OFF   ON  OFF 
    #define Z_R3 ((Z_Rsw2 + 0      + 0      + Z_Rout))     // OFF   ON   ON
    #define Z_R4 ((0      + Z_Rsw1 + Z_Rsw0 + Z_Rout))     //  ON  OFF  OFF
    #define Z_R5 ((0      + Z_Rsw1 + 0      + Z_Rout))     //  ON  OFF   ON
    #define Z_R6 ((0      + 0      + Z_Rsw0 + Z_Rout))     //  ON   ON  OFF
    #define Z_R7 ((0      + 0      + 0      + Z_Rout))     //  ON   ON   ON    
    
    // Rout : Rall (max , min)
    #define Z_R0max  (((Z_Rout * Z_gosaMax) / Z_R0))
    #define Z_R0min  (((Z_Rout * Z_gosaMin) / Z_R0))
    #define Z_R1max  (((Z_Rout * Z_gosaMax) / Z_R1))
    #define Z_R1min  (((Z_Rout * Z_gosaMin) / Z_R1))
    #define Z_R2max  (((Z_Rout * Z_gosaMax) / Z_R2))
    #define Z_R2min  (((Z_Rout * Z_gosaMin) / Z_R2))    
    #define Z_R3max  (((Z_Rout * Z_gosaMax) / Z_R3))
    #define Z_R3min  (((Z_Rout * Z_gosaMin) / Z_R3))    
    #define Z_R4max  (((Z_Rout * Z_gosaMax) / Z_R4))
    #define Z_R4min  (((Z_Rout * Z_gosaMin) / Z_R4))
    #define Z_R5max  (((Z_Rout * Z_gosaMax) / Z_R5))
    #define Z_R5min  (((Z_Rout * Z_gosaMin) / Z_R5))    
    #define Z_R6max  (((Z_Rout * Z_gosaMax) / Z_R6))
    #define Z_R6min  (((Z_Rout * Z_gosaMin) / Z_R6))   
    #define Z_R7max  (((Z_Rout * Z_gosaMax) / Z_R7))
    #define Z_R7min  (((Z_Rout * Z_gosaMin) / Z_R7))    
    
    // threshold lvevel
    //  GND(0[V]) -> 0_1 -> 1_2 -> 2_3 -> 3_4 -> 4_5 -> 5_6 -> 6_7 -> Vcc(3.3[V]) 
    //  --------------+------+------+------+------+------+------+---------------
    //   SW0 |    OFF     ON    OFF     ON    OFF    ON     OFF     ON     
    //   SW1 |    OFF    OFF     ON     ON    OFF   OFF      ON     ON
    //   SW2 |    OFF    OFF    OFF    OFF     ON    ON      ON     ON 
    
    #define Z_threshold0_1  (((Z_R0max + Z_R1min) / 2))
    #define Z_threshold1_2  (((Z_R1max + Z_R2min) / 2))    
    #define Z_threshold2_3  (((Z_R2max + Z_R3min) / 2))    
    #define Z_threshold3_4  (((Z_R3max + Z_R4min) / 2))        
    #define Z_threshold4_5  (((Z_R4max + Z_R5min) / 2))    
    #define Z_threshold5_6  (((Z_R5max + Z_R6min) / 2))    
    #define Z_threshold6_7  (((Z_R6max + Z_R7min) / 2))            
    
    void adInput(float ad, uint8_t swInNo);

};    

#endif    // _SWANALOG_H
