/* mbed Sound Library
 * Copyright (c) 2011 suupen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/***********************************************************************/
/*                                                                     */
/*    Sound.h                                                          */
/*                                                                     */
/*    V0.1 : 2012/xx/xx                                                */
/***********************************************************************/
#ifndef _SOUND_H
#define _SOUND_H

#include <string>
#include "types.h"


/** Sound output control class, based on a PwmOut
*
* Example:
* @code
* // Output tone and merody 
*
* #include "mbed.h"
* #include "Sound.h"       // sound library header
*
* Sound sound(p21, p10);      // 1tu me no sound syuturyoku (pwmOut = p21,  digitalOut = p10)
*
*
* <scematic>
*                                  
*                  --------      _/
*  mbed(p21) -----|R:200ohm|----| |  speaker
*       pwmOut     --------     | | 
*                               | |
*  mbed(p10) -------------------|_|
*       digitalOut                \
*
* //--------------------------------
* // "westminster chime" merody data
* //--------------------------------
*    const Sound::sound_t WESTMINSTER[] =     {
*    //   hanon siji 0:b(flat)   1:tujo  2:#(sharp)
*    //   |  C1 - B9 kan deno onkai(Gx ha 9x ni okikae te siji)  0xFF=end data
*    //   |  |   time (1/1[ms]/count)   
*    //   |  |   |    envelope(yoin) (1/1 [ms]/count)
*    //   |  |   |    |
*        {1,0xA4,1200,1000},
*        {1,0xF4,1200,1000},
*        {1,0x94,1200,1000},
*        {1,0xC4,2400,1000},
*
*        {1,0xC4,1200,1000},
*        {1,0x94,1200,1000},
*        {1,0xA4,1200,1000},
*        {1,0xF4,2400,1000},
*      
*        {1,0xFF,1000,0},    // end
*    };
*
* /--------------------------------------
* * main
* /---------------------------------------
* int main() {
*
* //---------------------
* // enso data no settei
* //---------------------
* // sound.sound_enso("/local/enso.txt");     // mbed local file data "enso.txt" load (sita ni data no rei wo oite oku)
*    sound.sound_enso((Sound::sound_t*)WESTMINSTER);
*    
*    //---------------------------------------------------
*    // output tone
*    //---------------------------------------------------
*    // tone1 
*    Sound::sound_t oto = {1,0x95,200,100};
*    sound.sound_sound(oto);
*    while(sound.sound_sound() == true){}
*            
*    // tone2
*    oto.hanon = 1; oto.onkai = 0xA5; oto.time = 2000; oto.envelope = 1000;
*    sound.sound_sound(oto);
*    while(sound.sound_sound() == true){}
*
*    //---------------
*    // output merody
*    //--------------
*    sound.sound_enso(true);
*
*    while(1) {
*    }
*}
*
* @endcode
*/














class Sound {
public:

/** tone data struct
*
* @param uint8_t hanon      :  hanon siji 0:b(flat)   1:tujo  2:#(sharp)
* @param uint16_t onkai     : C1 - B9 kan deno onkai(Gx ha 9x ni okikae te siji)    0x00:hatuon teisi    0xFF:enso syuryo
* @param uint16_t time      : hatuon jikan (1/1 [ms]/count)
* @param uint16_t envelope  : yoin(envelope) jikan (1/1 [ms]/count)
*/
typedef struct {
    uint8_t     hanon;      // hanon siji 0:b(flat)   1:tujo  2:#(sharp)
    uint16_t    onkai;      // C1 - B9 kan deno onkai(Gx ha 9x ni okikae te siji)
                            // 0x00:hatuon teisi    0xFF:enso syuryo
    uint16_t  time;         // hatuon jikan (1/1 [ms]/count)
    uint16_t  envelope;     // yoin(envelope) jikan (1/1 [ms]/count)
} sound_t;

/*
// hanon no menber
enum{
    Z_hanonF = 0,    // b
    Z_hanonN ,        // tujo
    Z_hanonS,        // #
};
*/

    /** Create a sound object connected to the specified PwmOut pin & DigitalOut pin
    *
    * @param pin PwmOut pin to connect to 
    * @param pin DigitalOut pin to connect to
    */
    Sound(PinName pwm , PinName kijun);

    /** Check tone
    *
    * @param return A bool ture : output 
    *      \\false: none
    */
    bool sound_sound(void);                     // oto no syuturyoku jotai wo kakunin
                                                //      true : oto ari     false : oto nasi
    /** output tone
    *
    * @param sound_t data : tone data set
    */
    void sound_sound(sound_t data);             // oto no syuturyoku
                                                //      para : oto no data
    
    /** merody data set (file data)
    *
    * @param merody data file path and name  (example : "/local/merodyFileName.txt")
    */
    bool sound_enso(char *path);                 // enso data wo file kara yomikomi
                                                 //     true : data kakunou OK    false: data kakunou NG
    /** merody data set (data table)
    *
    * @param merody data table name  (example : "(Sound::sound_t*)WESTMINSTER")
    */
    void sound_enso(Sound::sound_t* onpudata);   // enso data wo program no data table kara yomikomi

    /** check merody
    *
    * @param raturn ture:merody output  //false:merody stop
    */
    bool sound_enso(void);                       // enso jyotai check
                                                 //     true : enso chu    false : enso shuryo
    /** request merody start or stop
    *
    * @param ture:start merody //false:stop merody
    */
    void sound_enso(bool siji);                  // enso start / stop
                                                 //     true : enso start  false : enso stop
    
//protected:    
private:
    SPI _pwm;
    DigitalOut _kijun;
    
    Ticker sound_timer;
    Timer hatuon_jikan;


#define Z_pwmSyuuki (1)             // PWM syuuki           (1/1[us]/count)     syokichi = 1
#define Z_pwmBunkaino (16)          // PWM bunkaino         (1/1 [count]/count) 0:PWM=0% - 15:PWM=100% (SPI miso output bit(16bit)
#define Z_pulseCheckSyuuki (20)     // puls check syuuki    (1/1 [us]/count)    syokichi = 20 (10us ika deha pwm settei ga ijo ni naru)


typedef struct{
    uint32_t syuuki;    // oto no syuuki no count suu wo kioku (1/1 [us]/count)
    uint32_t jikan;     // oto no syuturyoku jikan wo kanri    (1/1 [us]/count)
    uint32_t envelope;  // envelope counter 0 - 1000000[us](1[s])
    uint32_t shokichienvelope;  // shokichi envelope counter 0 - 1000000[us](1[s])
} soundout_t;


soundout_t O_sound; // oto no syuturyoku pwm data

sound_t enso[100];      // mbed drive no enso data kakuno yo       
sound_t *ensoDataTable; // enso data no sento address kioku

    void sound_out(float siji, int8_t fugo);
    void pulseCheck(void);
        uint8_t F_pwmSet;    // 0:zenhan hansyuuki 1:kohan hansyuuki  wo request 
        uint32_t C_syuukiKeika;
        uint8_t f_muonSet;           // muon ji no sound_out() syori wo kurikaesu no wo fusegu
        uint32_t C_1msProcess;   // 1ms syuuki syori counter (Z_pulseCheckSyuuki/1 [us]/count))
        uint16_t D_SPIpwm;            // SPI pwm no output data pwm 0 = 0x0000, pwm 1/15 = 0x0001, pwm 2/15 = 0x0003, ... , pwm 14/15 = 0x7FFF, pwm 15/15 = 0xFFFF
    
    
    void sound_ensoSyori(void);
        int keikajikan;  // 1tu no oto no keikajikan
        sound_t* onpu; // onpu data no sento address

   
};    


#undef    _SOUND_C
#endif    // _SOUND_H
