/* mbed Eight Dot Matrix LED Library
 * Copyright (c) 2011 suupen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


//================================
//    EightDotMatrixLed.h        
// V1.0 : 2011/12/03                       
//                               
//================================
#ifndef _EIGHTDOTMATRIXLED_H
#define _EIGHTDOTMATRIXLED_H

/** Seven segment Numeric LED control class
 *
 * Example:
 * @code
 * //============================================
 * // eightDotMatrixLed Library Example
 * //
 * // This program is used to control the LED 32.
 * //
 * //
 * // <schematic>
 * //                               --|>|-- 
 * //                                A   K
 * //   seg A(p5) -----R(200[ohm])--- LED ----- com0(p13)
 * //               |
 * //               -- R(200[ohm])--- LED ----- com1(p14)
 * //               |
 * //               -- R(200[ohm])--- LED ----- com2(p28)
 * //               |
 * //               -- R(200[ohm])--- LED ----- com3(p27)
 * //
 * //
 * //   same : segB(p6), segC(p7), segD(p8), segE(p9), segF(p10), segG(p11), segH(p12)
 * //
 * // <Description of LED control>
 * // com0 Led off to on ,on to off smooth (lighting = 0 to 100 [%]
 * // com1 Led off to on smooth,   on to off hard
 * // com2 Led off to on, on to off smooth (lighting = 0 to gray data)
 * // com3 gray data movement
 * //
 * //=============================================
 *
 * #include "mbed.h"
 *
 * #include "EightDotMatrixLed.h"
 *
 * //                           common type (0:anode common 1:cathode common)
 * //                           |  
 * //                           |   segA segB segC segD segE segF segG segh com1 com2 com3 com4  (com5 to com8 = disable)                          
 * //                           |   |    |    |    |    |    |    |    |    |    |    |    | 
 * EightDotMatrixLed segmentled(1, p5,  p6,  p7,  p8,  p9, p10, p11, p12, p13, p14, p28, p27);
 *
 * // com0 Led off to on ,on to off smooth (lighting = 0 to 100 [%]
 * // com1 Led off to on smooth,   on to off hard
 * // com2 Led off to on, on to off smooth (lighting = 0 to gray data)
 * // com3 gray data movement
 *
 * uint8_t D_dotGrayData[4][8] =   {
 *                                     {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},   // com0 disable
 *                                     {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},   // com1 disable
 *                                     {0,      14,   29,   43,   57,   71,   86,  100},   // com2 enable (0 to 100 [%])
 *                                     {0,      14,   29,   43,   57,   71,   86,  100},   // com3 enable (0 to 100 [%])
 *                                  };
 * uint8_t D_dotDigitalData[4][8] = {
 *                                     {0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // com0 0:off, 1:on
 *                                     {0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},   // com1 0:disalbe 1:on                                    
 *                                     {0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},   // com2 0:off, 1:on
 *                                     {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}    // com3 disable
 *                                  };
 *
 * Timer timer;    // data change timer
 *   
 * int main() {
 *     uint8_t com, seg;
 *     uint8_t wk0, wk1, wk2, wk3;
 *     
 *     timer.start();
 *
 *     while(1) {
 *         // After 500[ms] to start the process
 *         if(timer.read_ms() >= 500){
 *             timer.reset();
 *
 *             // com0 Led off to on ,on to off smooth (lighting = 0 to 100 [%]
 *             wk0 = D_dotDigitalData[0][7];
 *             for(seg = 7; seg > 0; seg--){
 *                 D_dotDigitalData[0][seg] = D_dotDigitalData[0][seg - 1];
 *             }
 *             D_dotDigitalData[0][0] = wk0;
 *           
 *             // com1 Led off to on smooth,   on to off hard
 *             wk1 = D_dotDigitalData[1][7];
 *             for(seg = 7; seg > 0; seg--){
 *                 D_dotDigitalData[1][seg] = D_dotDigitalData[1][seg - 1];
 *             }
 *             D_dotDigitalData[1][0] = wk1;
 *            
 *             // com2 Led off to on, on to off smooth (lighting = 0 to gray data)
 *             wk2 = D_dotDigitalData[2][7];
 *             for(seg = 7; seg > 0; seg--){
 *                 D_dotDigitalData[2][seg] = D_dotDigitalData[2][seg - 1];
 *             }
 *             D_dotDigitalData[2][0] = wk2;
 *
 *             // com3 gray data movement
 *             wk3 = D_dotGrayData[3][7];
 *             for(seg = 7; seg > 0; seg--){
 *                 D_dotGrayData[3][seg] = D_dotGrayData[3][seg - 1];
 *             }
 *             D_dotGrayData[3][0] = wk3;
 *         }
 *        
 *         // This function, please repeat the process in less than 1ms.
 *         segmentled.EightDotMatrixLed_main((uint8_t*)D_dotGrayData, (uint8_t*)D_dotDigitalData);      
 * 
 *     }
 *}
 * @endcode
 */



#include "types.h"

class EightDotMatrixLed {
public:



    /** Create a eight dot matrix led array object connected to the specified DigitalOut pin
    * @param commonPole The polarity of the seven segment led common  0:Anode common, 1:Cathode common
    * @param seg_a - seg_h DigitalOut pin to connect to.    To provide members with an array of uint8_t digit minutes.
    * @param com_1 - com_8 DigitalOut pin to connect to.    To provide members with an array of uint8_t digit minutes. 8 digits maximum   
    */
    EightDotMatrixLed(uint8_t commonPole,
                PinName seg_a, PinName seg_b, PinName seg_c, PinName seg_d, PinName seg_e, PinName seg_f, PinName seg_g, PinName seg_h,
                PinName com_1 = NC, PinName com_2 = NC, PinName com_3 = NC, PinName com_4 = NC,
                PinName com_5 = NC, PinName com_6 = NC, PinName com_7 = NC, PinName com_8 = NC);
    
    /** Data set to the seven segment LED display
    * @param uint8_t* grayData[8][8]    address pointer : 0 - 100  (1/1 [%]/count) other:disable example 0:ledOff,  100:led max lighting 
    * @param uint8_t* digitalData[8][8] address pointer : 0 :ledOff   1 : ledOn  other:disable (1[s] de henka)
    */
    void EightDotMatrixLed_main(uint8_t* grayData, uint8_t* digitalData);

private:
void segmentGrayDataKosin(void);
void comAllClear(void);
void segAllClear(void);
void segDataSet(uint8_t keta);
void output(void);

//  pin set_seg, _com;
    DigitalOut _seg_a, _seg_b, _seg_c, _seg_d, _seg_e, _seg_f, _seg_g, _seg_h;
    DigitalOut _com_1, _com_2, _com_3, _com_4, _com_5, _com_6, _com_7, _com_8;
    
    Ticker timer;

#define Z_comSuu (8)   // max number of common pin
#define Z_segSuu  (8)   // max number of segment pin
#define Z_grayMax (100)  // grayData max 100 kaicho
#define Z_pwmGrayMax (100) // pwm max (led heno pwm syuturyoku no max)

uint8_t* A_dotDigitalData;  // D_dotDigitalData[Z_comSuu][Z_segSuu] address pointer
    //uint8_t D_dotDigitalData[Z_comSuu][Z_segSuu];  // segment dot digital data 0 :ledOff ,1 : ledOn  other:disable (1[s] de henka)
uint8_t* A_dotGrayData;     // D_dotGrayData[Z_comSuu][Z_segSuu] address pointer
    //uint8_t D_dotGrayData[Z_comSuu][Z_segSuu];  // setment dot gray data 0 - 100  (1/1 [%]/count) other:disable example 0:ledOff,  100:led max lighting

uint8_t DT_pwmGray[Z_grayMax + 1];  // gray data kara pwm data heno henkan table

uint8_t D_dotPwmData[Z_comSuu][Z_segSuu]; // pwm output data  0 - 100  (1/1 [%]/count) example 0:ledOff,  100:led max lighting

uint8_t D_comNull;  // comX Null check No set (0:all com is NC 1:com1 connect, 2:com2 connect,...,8:com8 connect(all com connect)

uint8_t D_commonOn; // common On level set   0:Anode common   1:Cathode common
uint8_t D_commonOff;

uint8_t D_segmentOn; // segment On level set 0:Cathode common 1:Anode common
uint8_t D_segmentOff;
};

#endif    // _EIGHTDOTMATRIXLED_H
