/**
 * Digital Switch Library
 *   include chattering cancel process.
 *
 *
 * <scematic>
 *
 *     +---------- mbed (pxx) (internal pullUp)
 *     |
 *     o | sw
 *       |
 *     o
 *     |
 *    -+- GND
 *
 * <timing chart>
 *
 *                 down Side        up Side
 *      interrupt  !                !
 *     off---------+                +-------------------
 *                 |                |  !
 *      on         +----------------+  !
 *                   !                 !
 *     edge          on edge           off edge
 *
 *
 * <example program>
 * @code
 * #include "mbed.h"
 * #include "DigitalSw.h"
 *
 * DigitalSw sw(p21);
 *
 * DigitalOut level(LED4);
 * DigitalOut onEdgeToggle(LED1);
 * DigitalOut offEdgeToggle(LED2);
 *
 * int main() {
 *   while(1) {
 *
 *       level = sw.getLevel();
 *
 *        if(sw.getOnEdge() == 1){
 *           onEdgeToggle = !onEdgeToggle;
 *           }
 *           
 *       if(sw.getOffEdge() == 1){
 *           offEdgeToggle = !offEdgeToggle;
 *           }
 *  }
 * }
 * @endcode
 */


#ifndef DIGITALSW_H
#define DIGITALSW_H

#include "mbed.h"

/**
 * digital switch.
 */


/**
 *  @class DigitalSw
 *  @brief nothine
 */
class DigitalSw
{

public:

    /** Create a DigitalSw port, connected to the specified sw pins
     *  @param sw pin
     *  @param (option) Register pull 1:PullUp(default), 0:PullDown
     *  @note  nothing
     */
    DigitalSw(PinName sw, uint8_t pull = 1);

    /**
     * Reset the edge and level.
     *       edge = off(0),  level = off(0)
     */
    void reset(void);

    /**
     * Get the number of level by the sw.
     *
     * @return Number of sw level. 0:Low(off)  1:Hi(on)
     */
    int16_t getLevel(void);

    /**
     * Get the number of on edge by the sw.
     *
     * @return Number of sw on edge. 0:no edge, 1:edge
     */
    int16_t getOnEdge(void);

    /**
     * Get the number of off edge by the sw.
     *
     * @return Number of sw off edge. 0:no edge, 1:edge
     */
    int16_t getOffEdge(void);


private:

    void swLevelUp(void);
    void swLevelDown(void);


    InterruptIn _sw;
    int8_t _pull;

    volatile int16_t _onEdge;
    volatile int16_t _offEdge;
    volatile int16_t _levelNow;
    volatile int16_t _levelOld;

};

#endif /* DIGITALSW_H */
