/* mbed Dc motor direct drive Library
 * Copyright (c) 2011 suupen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


//================================
//    DcMotorDirectDrive.h        
// V1.0 : 2011/12/05                       
//                               
//================================
#ifndef _DCMOTORDIRECTDRIVE_H
#define _DCMOTORDIRECTDRIVE_H

/** Dc motor direct drive class
 *
 * Example:
 * @code
 * //============================================
 * // DcMotorDirectDrive Library Example
 * //
 * // This program controls the motor 3.
 * // Repeat the forward and reverse motor.
 * // Continuously varies the rotational speed of the motor.
 * //
 * // <schematic>
 * //
 * //                                   motor1,2,3 : mabuchi motor RF-300C-11440  DC3V 18mA 2200rpm)
 * //                              __
 * //    p21(pwmOut)     ---------|  |  motor1
 * //                             |  |--
 * //    p22(digitalOut) ---------|__|
 * //
 * //                              __
 * //    p23(pwmOut)     ---------|  |  motor2
 * //                             |  |--
 * //    p24(digitalOut) ---------|__|
 * //
 * //                              __
 * //    p25(pwmOut)     ---------|  |  motor3
 * //                             |  |--
 * //    p26(digitalOut) ---------|__|
 * //
 * //=============================================
 *
 * #include "mbed.h"
 *
 * #include "DcMotorDirectDrive.h"
 *
 * //                          pwm pin
 * //                          |  
 * //                          |    reference pin                          
 * //                          |    |
 * DcMotorDirectDrive motor1(p21, p22);
 * DcMotorDirectDrive motor2(p23, p24);
 * DcMotorDirectDrive motor3(p25, p26);
 *
 *
 * DigitalOut led1(LED1);
 * DigitalOut led4(LED4);
 *
 * Timer timer;    // data change timer
 *    
 * int main() {
 *    float siji = 1.0f;  // motor output (+1.0 -> ... -> +0.5 -> -1.0 -> ... -> -0.5 -> +1.0)
 *    uint8_t fugo = 0;   // 0 : normal rotation,  1 : reverse rotation
 *    
 *    
 *    timer.start();
 *
 *    while(1) {
 *        // After 100[ms] to start the process
 *        if(timer.read_ms() >= 100){
 *            timer.reset();
 *            
 *            if(fugo == 0){
 *                // Normal rotation
 *                // siji = +1.0 -> +0.5 
 *                 siji -= 0.005;
 *                if(siji < 0.5f){
 *                    fugo = 1;
 *                    siji = -1.0f;
 *                }
 *            }
 *            else{
 *                // Reverse rotation
 *                siji += 0.005;
 *                if(siji > -0.5f){
 *                    fugo = 0;
 *                    siji = +1.0f;
 *                }
 *           }
 *            // Outputs a control instruction to the motor
 *            motor1.DcMotorDirectDrive_main(siji);           
 *            motor2.DcMotorDirectDrive_main(siji);
 *            motor3.DcMotorDirectDrive_main(siji);
 *        
 *            // rotation display
 *            if(siji > 0.5){
 *                // normal rotation
 *                led1 = 1;
 *                led4 = 0;
 *            }
 *            else{
 *                // reverse rotation
 *                led1 = 0;
 *             led4 = 1;        
 *            }
 *        }
 *    }
 * }
 * @endcode
 */



#include "types.h"

class DcMotorDirectDrive {
public:
    
    /** Create a eight dot matrix led array object connected to the specified DigitalOut pin
    * @param pwmPin PwmOut pin to connect to.           pwm pin (p21 - p26) 
    * @param referencePin DigitalOut pin to connect to.    Reference power supply pin (p5 - p30)   
    */

    DcMotorDirectDrive(PinName pwmPin, PinName referencePin);
    
    
    /** Data set to the seven segment LED display
    * @param float  request motor output request (-1.0 - 0.0 - 1.0)  -1.0 to 0.0 : Reverse rotation, 0.0 : stop, 0.0 to  +1.0 : normal rotation
    */
    void DcMotorDirectDrive_main(float request);

private:
    PwmOut _pwmPin;
    DigitalOut _referencePin;
    
    #define Z_pwmPeriod (10000) // pwm period (1/1 [us]/count) 10[ms](0.1[kHz])
    #define Z_stopRange (0.0) // pwm request stop range (-0.0 - +0.0)


};

#endif    // _DCMOTORDIRECTDRIVE_H
