#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "ExperimentServer.h"
#include "QEI.h"
#include "BezierCurve.h"
#include "MotorShield.h" 
#include "HardwareSetup.h"

#define BEZIER_ORDER_FOOT    7
#define BEZIER_ORDER_TORQUE    2 /// if x -> plots x+1 points, so should have x+1 Bezier points for T1 and T2 in MATLAB
//#define NUM_INPUTS (14 + 2*(BEZIER_ORDER_FOOT+1))
#define NUM_INPUTS (16 + 2*(BEZIER_ORDER_TORQUE+1))
#define NUM_OUTPUTS 21

#define PULSE_TO_RAD (2.0f*3.14159f / 1200.0f)

// Initializations
Serial pc(USBTX, USBRX);    // USB Serial Terminal
ExperimentServer server;    // Object that lets us communicate with MATLAB
Timer t;                    // Timer to measure elapsed time of experiment

QEI encoderA(PE_9, PE_11, NC, 1200, QEI::X4_ENCODING);  // MOTOR A ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderB(PA_5, PB_3, NC, 1200, QEI::X4_ENCODING);  // MOTOR B ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderC(PC_6, PC_7, NC, 1200, QEI::X4_ENCODING);  // MOTOR C ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderD(PD_12, PD_13, NC, 1200, QEI::X4_ENCODING);// MOTOR D ENCODER (no index, 1200 counts/rev, Quadrature encoding)

MotorShield motorShield(12000); //initialize the motor shield with a period of 12000 ticks or ~20kHZ
Ticker currentLoop;

// Variables for q1
float current1;
float current_des1 = 0;
float prev_current_des1 = 0;
float current_int1 = 0;
float angle1;
float angle_des1;
float velocity1;
float velocity_des1;
float duty_cycle1;
float angle1_init;
float shoulder;

// Variables for q2
float current2;
float current_des2 = 0;
float prev_current_des2 = 0;
float current_int2 = 0;
float angle2;
float angle_des2;
float velocity2;
float velocity_des2;
float duty_cycle2;
float angle2_init;
float elbow;

float l1;
float l2;

// Timing parameters
float current_control_period_us = 200.0f;     // 5kHz current control loop
float impedance_control_period_us = 1000.0f;  // 1kHz impedance control loop
float pre_buffer_time, traj_period, post_buffer_time;

// Control parameters
float current_Kp = 4.0f;         
float current_Ki = 0.4f;           
float current_int_max = 3.0f;       
float duty_max;      
float K_xx;
float K_yy;
float K_xy;
float D_xx;
float D_xy;
float D_yy;

// Model parameters
float supply_voltage = 12;     // motor supply voltage
float R = 2.0f;                // motor resistance
float k_t = 0.18f;             // motor torque constant
float nu = 0.0005;             // motor viscous friction

// Current control interrupt function
void CurrentLoop()
{
    // This loop sets the motor voltage commands using PI current controllers with feedforward terms.
    
    //use the motor shield as follows:
    //motorShield.motorAWrite(DUTY CYCLE, DIRECTION), DIRECTION = 0 is forward, DIRECTION =1 is backwards.
        
    current1 = -(((float(motorShield.readCurrentA())/65536.0f)*30.0f)-15.0f);           // measure current DO WE NEED TO ADJUST GEAR RATIO HERE?
    velocity1 = encoderA.getVelocity() * PULSE_TO_RAD;                                  // measure velocity        
    float err_c1 = current_des1 - current1;                                             // current errror
    current_int1 += err_c1;                                                             // integrate error
    current_int1 = fmaxf( fminf(current_int1, current_int_max), -current_int_max);      // anti-windup
    float ff1 = R*current_des1 + k_t*velocity1;                                         // feedforward terms
    duty_cycle1 = (ff1 + current_Kp*err_c1 + current_Ki*current_int1)/supply_voltage;   // PI current controller
    
    float absDuty1 = abs(duty_cycle1);
    if (absDuty1 > duty_max) {
        duty_cycle1 *= duty_max / absDuty1;
        absDuty1 = duty_max;
    }    
    if (duty_cycle1 < 0) { // backwards
        motorShield.motorAWrite(absDuty1, 1);
    } else { // forwards
        motorShield.motorAWrite(absDuty1, 0);
    }             
    prev_current_des1 = current_des1; 
    
    current2     = -(((float(motorShield.readCurrentB())/65536.0f)*30.0f)-15.0f);       // measure current 
    velocity2 = encoderB.getVelocity() * PULSE_TO_RAD;                                  // measure velocity  
    float err_c2 = current_des2 - current2;                                             // current error
    current_int2 += err_c2;                                                             // integrate error
    current_int2 = fmaxf( fminf(current_int2, current_int_max), -current_int_max);      // anti-windup   
    float ff2 = R*current_des2 + k_t*velocity2;                                         // feedforward terms
    duty_cycle2 = (ff2 + current_Kp*err_c2 + current_Ki*current_int2)/supply_voltage;   // PI current controller
    
    float absDuty2 = abs(duty_cycle2);
    if (absDuty2 > duty_max) {
        duty_cycle2 *= duty_max / absDuty2;
        absDuty2 = duty_max;
    }    
    if (duty_cycle2 < 0) { // backwards
        motorShield.motorBWrite(absDuty2, 1);
    } else { // forwards
        motorShield.motorBWrite(absDuty2, 0);
    }             
    prev_current_des2 = current_des2; 
    
}

int main (void)
{
    
    // Object for 7th order Cartesian foot trajectory
    BezierCurve rDesFoot_bez(2,BEZIER_ORDER_FOOT);
    BezierCurve torque_bez(2,BEZIER_ORDER_TORQUE);
    
    // Link the terminal with our server and start it up
    server.attachTerminal(pc);
    server.init();
    
    // Continually get input from MATLAB and run experiments
    float input_params[NUM_INPUTS];
    pc.printf("%f",input_params[0]);
    
    while(1) {
        
        // If there are new inputs, this code will run
        if (server.getParams(input_params,NUM_INPUTS)) {
            
                        
            // Get inputs from MATLAB          
            pre_buffer_time             = input_params[0];    // First buffer time, before trajectory
            traj_period                 = input_params[1];    // Trajectory time/length
            post_buffer_time            = input_params[2];    // Second buffer time, after trajectory
    
            angle1_init                 = input_params[3];    // Initial angle for q1 (rad)
            angle2_init                 = input_params[4];    // Initial angle for q2 (rad)
            shoulder                    = input_params[5];
            elbow                       = input_params[6];

            K_xx                        = input_params[7];    // Foot stiffness N/m
            K_yy                        = input_params[8];    // Foot stiffness N/m
            K_xy                        = input_params[9];    // Foot stiffness N/m
            D_xx                        = input_params[10];    // Foot damping N/(m/s)
            D_yy                        = input_params[11];    // Foot damping N/(m/s)
            D_xy                        = input_params[12];   // Foot damping N/(m/s)
            duty_max                    = input_params[13];   // Maximum duty factor
            l1                          = input_params[14];   // Length of 1st link
            l2                          = input_params[15];   // Length of 2nd link
          
            // Get foot trajectory points
            float foot_pts[2*(BEZIER_ORDER_FOOT+1)];
            for(int i = 0; i<2*(BEZIER_ORDER_FOOT+1);i++) {
              foot_pts[i]               = input_params[16+i]; // Bezier curve points
            }
            rDesFoot_bez.setPoints(foot_pts);
            
            // Get torque traj points
            float torque_pts[2*(BEZIER_ORDER_FOOT+1)];
            for(int i = 0; i<2*(BEZIER_ORDER_FOOT+1);i++) {
              torque_pts[i] = input_params[16+i];    
            }
            torque_bez.setPoints(torque_pts);
            float torque_des[2];
            
            // Attach current loop interrupt
            currentLoop.attach_us(CurrentLoop,current_control_period_us);
                        
            // Setup experiment
            t.reset();
            t.start();
            encoderA.reset();
            encoderB.reset();
            encoderC.reset();
            encoderD.reset();

            motorShield.motorAWrite(0, 0); //turn motor A off
            motorShield.motorBWrite(0, 0); //turn motor B off
                         
            // Run experiment
            while( t.read() < pre_buffer_time + traj_period + post_buffer_time) { 
                 
                // Read encoders to get motor states
                angle1 = encoderA.getPulses() *PULSE_TO_RAD + angle1_init;       
                velocity1 = encoderA.getVelocity() * PULSE_TO_RAD;
                 
                angle2 = encoderB.getPulses() * PULSE_TO_RAD + angle2_init;       
                velocity2 = encoderB.getVelocity() * PULSE_TO_RAD;           
                
                const float th1 = angle1;
                const float th2 = angle2;
                const float dth1= velocity1;
                const float dth2= velocity2;
 
                // Calculate the Jacobian
                float Jx_th1 = l1*cos(th1)+l2*cos(th1+th2);
                float Jx_th2 = l2*cos(th1+th2);
                float Jy_th1 = l1*sin(th1)+l2*sin(th1+th2);
                float Jy_th2 = l2*sin(th1+th2);
                

                // Calculate the forward kinematics (position and velocity) 
                float xFoot = l2*sin(th1+th2) + l1*sin(th1);
                float yFoot = -l2*cos(th1+th2) - l1*cos(th1);
                float dxFoot = l1*cos(th1)*dth1 + l2*cos(th1+th2)*(dth1+dth2);
                float dyFoot = l1*sin(th1)*dth1 + l2*sin(th1+th2)*(dth1+dth2);
                
                // Set gains based on buffer and traj times, then calculate desired x,y from Bezier trajectory at current time if necessary
                float teff  = 0;
                float vMult = 0;
                if( t < pre_buffer_time) { // impedance operational space to start point
                
                    teff = 0.f;
                    float th1_d= shoulder;
                    float th2_d= -elbow;
                    float e_th1= th1_d-th1;
                    float e_th2= th2_d-th2;
                    
                    float e_dth1= -dth1;
                    float e_dth2= -dth2;
                    
                    torque_des[0] = K_xx*e_th1 + D_xx*e_dth1;
                    torque_des[1] = K_yy*e_th2 + D_yy*e_dth2;
           
                    current_des1 = torque_des[0]/k_t;          
                    current_des2 = torque_des[1]/k_t;
                }
                else if (t < pre_buffer_time + traj_period) // torque control to follow Bezier
                {
                    K_xx = input_params[7];  // Foot stiffness N/m
                    K_yy = input_params[8];  // Foot stiffness N/m
                    K_xy = input_params[9];  // Foot stiffness N/m
                    D_xx = input_params[10];  // Foot damping N/(m/s)
                    D_yy = input_params[11];  // Foot damping N/(m/s)
                    D_xy = input_params[12]; // Foot damping N/(m/s)
                    
                    teff = (t-pre_buffer_time);
                    vMult = 1;
                    torque_bez.evaluate(teff/traj_period,torque_des);
                    current_des1 = torque_des[0]/k_t;          
                    current_des2 = torque_des[1]/k_t;
                }
                else // impedance operational space to end point
                {
                    teff = 0.f;
                    float th1_d= shoulder;
                    float th2_d= -elbow;
                    float e_th1= th1_d-th1;
                    float e_th2= th2_d-th2;
                    
                    float e_dth1= -dth1;
                    float e_dth2= -dth2;
                    
                    torque_des[0] = (K_xx*e_th1 + D_xx*e_dth1)/2;
                    torque_des[1] = (K_yy*e_th2 + D_yy*e_dth2)/2;
           
                    current_des1 = torque_des[0]/k_t;          
                    current_des2 = torque_des[1]/k_t;
                    
                }
                  
                // Form output to send to MATLAB     
                float output_data[NUM_OUTPUTS];
                // current time
                output_data[0] = t.read();
                // motor 1 state
                output_data[1] = angle1;
                output_data[2] = velocity1;  
                output_data[3] = current1;
                output_data[4] = current_des1;
                output_data[5] = duty_cycle1;
                // motor 2 state
                output_data[6] = angle2;
                output_data[7] = velocity2;
                output_data[8] = current2;
                output_data[9] = current_des2;
                output_data[10]= duty_cycle2;
                // foot state
                output_data[11] = xFoot;
                output_data[12] = yFoot;
                output_data[13] = dxFoot;
                output_data[14] = dyFoot;
                output_data[15] = current1*k_t;
                output_data[16] = current2*k_t;
                output_data[17] = torque_des[0];
                output_data[18] = torque_des[1];
                
                // Send data to MATLAB
                server.sendData(output_data,NUM_OUTPUTS);

                wait_us(impedance_control_period_us);   
            }
            
            // Cleanup after experiment
            server.setExperimentComplete();
            currentLoop.detach();
            motorShield.motorAWrite(0, 0); //turn motor A off
            motorShield.motorBWrite(0, 0); //turn motor B off
        
        } // end if
        
    } // end while
    
} // end main