#ifndef _BENCH02_H_
#define _BENCH02_H_
#include "MODSERIAL.h"

#include "mbed.h"
#include <string>

#include "SDFileSystem.h"
#include "constants.h"

#include "as5048.h"
#include "lcm101.h"
#include "SPTEPressureSensor.h"
#include "Valve.h"
#include "LinearActuator.h"

/**
 * Class to read out sensory information from the second test bench:
 * the knee joint angle as measured by an AMS AS5048 absolute rotary sensor
 * the vertical force applied externally at the hip joint
 * the pressure (for example of an actuator used in the device under test).
 * Additionally, an on /off valve can pressurise the actuator on the leg
 */
class Bench {
public:
    
    static const float kCutOffDegrees = 180.0f;
    static const float kCutOffRadians = 3.14159265359f;
    
    enum Joint {
        TOES,
        ANKLE,
        KNEE,
        HIP
    };
    
    enum DACChannel {
        A,
        B
    };
    
    // constructor
    Bench(PinName mosi = AS5048_MOSI, PinName miso = AS5048_MISO, PinName sck = AS5048_SCLK, PinName cs = AS5048_CS, 
        bool use5kN = sensors::use5kN,PinName p_lcm101 = LCM101, 
        PinName p_spte0 = SPTE_0, PinName p_spte1 = SPTE_1,PinName p_valve = VALVE_PIN,PinName p_laDir = LINACT_DIR_PIN, PinName p_laPwm = LINACT_PWM_PIN,
        PinName sd_mosi = SD_MOSI, PinName sd_miso = SD_MISO, PinName sd_sck = SD_SCK, PinName sd_cs = SD_CS,
        PinName tx = USBTX, PinName rx = USBRX);  

    
    //setup
    void initialise();
    
    void setLoggingFrequency(float logHertz);
    void setExtraColumns(string extraColumnNames[], int numCols);
    void setExtraData(float data[]);
    void update(); 

    bool isLogging();//indicates if we are now datalogging
    void stopLogging();//halts the data logging if we are now datalogging
    void StopLogging(); //halts the data logging if we are now datalogging
    void StartLogging(const char * fname_append = "data"); //starts datalogging
    void RestartLogging(const char * fname_append = "data");//restart datalogging
    void LogData();
    
    //reading angle sensors
    As5048* get_as5048();
    
    float getDegrees(int i_joint);
    float getDegrees(Joint joint);

    float getRadians(int i_joint);
    float getRadians(Joint joint);

    const char* getJointName(int i_joint);
    const char* getJointName(Joint joint);

    //reading and calibrating force and pressure sensors
    float getForce();
    void nullForce();
    
    float getPressure0();
    void nullPressure0();
    float getPressure1();
    void nullPressure1();
    
    //MAX5322 DACs (two channels per DAC)
    void setDACA(DACChannel ch, unsigned int bitValue);
    void setDACB(DACChannel ch, unsigned int bitValue);
    
    //controlling valve
    bool getValve();
    void setValve(bool pressurise); 
    
    // controlling linear actuator
    void setDir(bool dir);
    bool getDir();
    void setPWM(int pwm);    
    
    /** Object for printing/receiving data to the PC the test rig is connected 
    * to. Note that it uses the MODSERIAL.h class.
    */
    MODSERIAL pc;
    
    void pausePrint();
    void resumePrint();
    
private:
    //timing
    int loggingUS; //microseconds between recording data logging
    float loggingHz;// Hz of data logging
    
    Ticker tick_update, tick_serial, tick_logging; //for the updating of sensor values, printing of serial data and logging of data
    Timer timer;
    Timer keyIntT;//timer to debounce keypresses
    int keyIntOffset;//for setting initial value of keyIntT
    int firstReadMS; //first timer value read
    bool startedLogging; //in the middle of a logging cycle
    
    bool is_logging;
    bool is_printing;
    bool was_printing; //if printing is paused true if is_printing was true
    char k; //value from keyboard
            
    //joint angle 
    As5048 as5048_;
    
    //load cells
    Lcm101 loadCell5kN;
    Lcm101 loadCell1kN;
    bool use5kN;//determine which load cell is used
    
    //pressure sensors
    SPTEPressureSensor spte0;
    SPTEPressureSensor spte1;
    
    //valve
    ValveDigital valveFesto;
    
    //linear actuator
    LinearActuator LinAct;
    
    //SD card
    SDFileSystem sd;
    FILE * fp_data;
    int fname_prepend;//file number
    bool sd_card_present;//card detected    
       
    static const int maxCols = 5;
    int usedExtraCols; //number of extra columns useds
    string extraColNames[maxCols];//names of extra columns (up to maxCols allowed with 15 characters)  
    float extraColValues[maxCols];//value held in extra columns for current time step
       
    void InitSdCard();      

    //for toggling logging or printing
    void ToggleState(MODSERIAL_IRQ_INFO *q);

    void ToggleLogging();
    
    // serial printing
    void TogglePrinting();
    void PrintStatus();
    void PrintMenu();  
};
#endif