/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "BLEDevice.h"
#include "HeartRateSecSampleSrv.h"
#include "DeviceInformationService.h"

/* Enable the following if you need to throttle the connection interval. This has
 * the effect of reducing energy consumption after a connection is made;
 * particularly for applications where the central may want a fast connection
 * interval.*/
#define UPDATE_PARAMS_FOR_LONGER_CONNECTION_INTERVAL 0

BLEDevice  ble;
DigitalOut led1(LED1);

const static char     DEVICE_NAME[]        = "HRM_SEC";
static const uint16_t uuid16_list[]        = {GattService::UUID_HEART_RATE_SERVICE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE};
static volatile bool  triggerSensorPolling = false;

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    printf("Disconnected!\r\n");
    ble.startAdvertising(); // restart advertising
}


void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */

    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread. */
    triggerSensorPolling = true;
}


void connectionCallback(Gap::Handle_t handle, Gap::addr_type_t peerAddrType, const Gap::Address_t peerAddr,
                                              Gap::addr_type_t ownAddrType,  const Gap::Address_t ownAddr,
                                              const Gap::ConnectionParams_t * pParam)
{
    printf("Connected!\r\n");
}


void passkeyDisplayCallback(Gap::Handle_t handle, const Gap::Passkey_t passkey)
{
    uint8_t i;
    
    printf("Input passKey: ");
    for(i = 0; i < 6; i++) {
        printf("%c ", passkey[i]);
    }
    printf("\r\n");
}


void securitySetupCompletedCallback(Gap::Handle_t handle, Gap::SecurityCompletionStatus_t status)
{
    if(!status) {
        printf("Security success\r\n", status);
    } else {
        printf("Security failed\r\n", status);
    }
}


int main(void)
{
    led1 = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, 1); // blink LED every second
    printf("Started\r\n");
    
    /* Initialize BLE module */
    ble.init();
    
    /* Initialize BLE security */
    ble.initializeSecurity(true, true, Gap::IO_CAPS_DISPLAY_ONLY, NULL);
    
    /* Set callback functions */
    ble.onConnection(connectionCallback);
    ble.onDisconnection(disconnectionCallback);
    ble.onPasskeyDisplay(passkeyDisplayCallback);
    ble.onSecuritySetupCompleted(securitySetupCompletedCallback);
    
    /* Setup primary service. */
    uint8_t hrmCounter = 100; // init HRM to 100bps
    HeartRateSecService hrService(ble, hrmCounter, HeartRateSecService::LOCATION_FINGER);

    /* Setup auxiliary service. */
    DeviceInformationService deviceInfo(ble, "ARM", "Model1", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");

    /* Setup advertising. */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_HEART_RATE_SENSOR);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.setAdvertisingInterval(1000);
    
    ble.startAdvertising();

    // infinite loop
    while (1) {
        // check for trigger from periodicCallback()
        if (triggerSensorPolling && ble.getGapState().connected) {
            triggerSensorPolling = false;

            // Do blocking calls or whatever is necessary for sensor polling.
            // In our case, we simply update the HRM measurement. 
            hrmCounter++;
            
            //  100 <= HRM bps <=175
            if (hrmCounter == 175) {
                hrmCounter = 100;
            }
            
            // update bps
            hrService.updateHeartRate(hrmCounter);
        } else {
            ble.waitForEvent(); // low power wait for event
        }
    }
}
