#include "mbed.h"
#include "main.h"
/* This code is to drive the Measurement Specialties MS5611-01BA03
 * pressure sensor. The sensor measures pressure via piezo electric
 * sensor, that is measured by a 24bit delta-sigma ADC. Temperure is
 * also measured via the same ADC.
 * 
 * Pressure is then calculated via a first order equation in this code
 * where both the pressure offset and linear term are also both linear
 * dependent on temperature.
 * 
 * Higher order terms are avaiable, but are not included in this code
 * at present.
 *
 * The sensor can be accessed by either SPI or I2C, this code uses SPI.
 *
 * The algorithms are based on the MS5611-01BA03 data sheet, and also
 * AN520
 *
 * Some details are not clear in the documents, these are:
 * . The commands are 7bit, the 8th bit is a stop bit and must be zero
 * . The sensor will reply to the command with 0xfe, e.g. has it own stop bit
 * . If 0xfe is not receieved error condition has happened
 * . This starts the command
 * . Chip Select (CS) must be pulled high (inactive) after every command
 * . This can either be immediatly after issuing the command
 * . or after any delay needed for the command to execute
 * . The linear cooeficients are held in the PROM
 * . This has a CRC4 check sum
 * . The CRC4 check sum is based on the polymonial 0x13=x^4+x+1
 * . The CRC4 acts on the first 7 PROM values each 16bit,
 * . The 8th PROM value, the CRC acts on the 12 MSB of the register
 * . In addition it requires bits 9-12 to be zeroed in the check sum  
 */

Serial pc(SERIAL_TX, SERIAL_RX, 115200); // tx, rx
//DigitalOut led(LED_RED);
//SPI spi(PTC6,PTC7,PTD1,PTC0);
//SPI spi(PTC6,PTC7,PTD1); // my prefered, but mixed
SPI spi(PTC6,PTC7,PTC5); // PTC5 - the LED
//SPI spi(PTD2,PTD3,PTD1);
DigitalOut cs(PTC0);

int main()
{
    uint16_t PROM[8];
    uint32_t D1,D2;
    double T,P;
    cs.write(1); // disable all SPI
    pc.printf("Hello World!\n");
    MS5611_init(PROM);
    for (int i=0; i<8; i++) {
        pc.printf("Prom(%i): %i\n",i,PROM[i]);
    }
    
    while (1) {
    // the third paramater gives over samping
    // 0 - 256
    // 1 - 512
    // 2 - 1024
    // 3 - 2048
    // 4 - 4096
    MS5611(&D1,&D2,4);
//    pc.printf("%i\t%i\t",D1,D2);

    MS5611_phys(D1,D2,PROM,&T,&P);

//    pc.printf("%f\t",T/100.0);
//    pc.printf("%f\r\n",P/100.0);
    pc.printf("%i\t%i\t%f\t%f\n",D1,D2,T/100.0,P/100.0);
    wait(1.0);
    }
}

void MS5611_init(uint16_t *PROM)
{
    uint32_t crc=0;
    spi.format(8,0); // 8 bit mode 0 - this is default anyway but good to code.
    spi.frequency(1000000); // 1MHz is this the minimum?
    cs.write(0); // Enable the MS5611 intercae
    if (spi.write(0x1e)!=0xfe) { // Reset the device
        pc.printf("Error reseting the device\n");
    }
    wait_ms(3); // give time for ROM to reload 2.8ms by the spec
    cs.write(1); // close the connection - does it finish the command?
    wait_us(10); // Pause after putting CSB high
    for (int i=0; i<8; i++) {
        cs.write(0); // enable the SC to start a command.
        if (spi.write(0xa0|(i<<1))!=0xfe) {
            pc.printf("Error reading prom(%i)\n",i);
        }
        PROM[i]=((uint16_t) spi.write(0x00))<<8;
        PROM[i]|=((uint16_t) spi.write(0x00));
        cs.write(1); // disable CS to finish the command
        crc|=(i==7)?(PROM[i]&0xff00):PROM[i]; // Note AN520 gives this bit mask
        for (int j=0; j<16; j++) {
            crc=crc<<1;
            if (crc&0x1000000) { // implimt the CRC4 algorithm 0x13 -
                crc^=0x1300000; // This won't do the last 4 bits of PROM7
            }
        }
        wait_us(10); // Pause probably not stricly needed, but after switching CSB high give a bit of time.
    }
    if ((crc>>20)!=(PROM[7]&0xf))
        pc.printf("CRC check sum: %x vs recorded %x\n",crc>>20,PROM[7]&0xf);
}

void MS5611(uint32_t *D1, uint32_t *D2, int os)
{
    //  cs.write(0);
    spi.format(8,0);
    spi.frequency(1000000);
    cs.write(0); // Enable the MS5611 intercae
    if (spi.write(0x1e)!=0xfe) { // and reset the device
        pc.printf("Error reseting the device\n");
    }
    wait_ms(3); // give time for ROM to reload 2.8ms by the spec
    cs.write(1); // close the connection - does it finish the command?
    wait_us(10); // Pause after putting CSB high
    cs.write(0); // Enable the MS5611 intercae
    if (spi.write(0x40|(os<<1))!=0xfe) { // D1 please
        pc.printf("Error asking for D1\n");
    }
    cs.write(1); // Disable the MS5611 intercae
    wait_us(600<<os); // pause for read, longer when oversampling
    cs.write(0); // Enable the MS5611 intercae
    if(spi.write(0x00)!=0xfe) { // can I have the result?;
        pc.printf("Error reading D1\n");
    }
    *D1=((uint32_t) spi.write(0x00))<<16;
    *D1|=((uint32_t) spi.write(0x00))<<8;
    *D1|=((uint32_t) spi.write(0x00));
    cs.write(1); // and terminate the command
    wait_us(10); // Pause after putting CSB high
    cs.write(0); // Enable the MS5611 intercae
    if (spi.write(0x50|(os<<1))!=0xfe) { // D2 please
        pc.printf("Error asking for D2\n");
    }
    cs.write(1); // Disable the MS5611 intercae
    wait_us(600<<os); // pause for read, longer when oversampling
    cs.write(0); // Enable the MS5611 intercae
    if(spi.write(0x00)!=0xfe) { // can I have the result?;
        pc.printf("Error reading D2\n");
    }
    *D2=((uint32_t) spi.write(0x00))<<16;
    *D2|=((uint32_t) spi.write(0x00))<<8;
    *D2|=((uint32_t) spi.write(0x00));
    cs.write(1); // and terminate the command
}

void MS5611_phys(uint32_t D1,uint32_t D2,uint16_t *PROM, double *T, double *P)
{
    int64_t dt=((int64_t) D2)-(((int64_t) PROM[5])<<8);
    *T=2000.0+((double) dt*PROM[6])/8388608.0;
    int64_t off128=(((int64_t) PROM[2])<<23)+(((int64_t) PROM[4])*dt);
    int64_t sens256=(((int64_t) PROM[1])<<23)+(((int64_t) PROM[3])*dt);
    *P=(((double) D1)*((double) sens256)/4194304.0-((double) off128))/4194304.0;
/* this keeps the power expansion - result not diffferent from above
  Hence the above doesn't have rouding problems.
    Pp[0]=
    -((double) PROM[2])*2.0+((double) PROM[4])*((double) PROM[5])/16384.0
    +((double) D1)*( ((double) PROM[1])*32768.0
        -((double) PROM[3])*((double) PROM[5]))/68719476736.0
    -((double) PROM[4])*((double) D2)/4194304.0
    +((double) D1)*((double) PROM[3])*((double) D2)/17592186044416.0;
 * this breaks it down by terms
    Pp[0]=-((double) PROM[2])*2.0+((double) PROM[4])*((double) PROM[5])/16384.0;
    Pp[1]=+((double) D1)*( ((double) PROM[1])*32768.0
        -((double) PROM[3])*((double) PROM[5]))/68719476736.0;
    Pp[2]=-((double) PROM[4])*((double) D2)/4194304.0;
    Pp[3]=+((double) D1)*((double) PROM[3])*((double) D2)/17592186044416.0;
*/

}