#ifndef RINGBUFFER_H
#define RINGBUFFER_H

#include <stdint.h>

#define RX_BUFF_SIZE            129 
#define BYTES_PER_INTERRUPT     1

typedef struct {
    uint8_t buffer[RX_BUFF_SIZE];
    uint16_t length;
    uint16_t start;
    uint16_t end;
} RingBuffer;

void RingBuffer_init(RingBuffer *buffer);

int16_t RingBuffer_read(RingBuffer *buffer, uint8_t *target, uint16_t amount);
int16_t RingBuffer_write(RingBuffer *buffer, uint8_t *data, uint16_t amount);
bool RingBuffer_readbyte(RingBuffer *buffer, uint8_t *target);
bool RingBuffer_writebyte(RingBuffer *buffer, uint8_t byte);

bool RingBuffer_isEmpty(RingBuffer *buffer);
bool RingBuffer_isFull(RingBuffer *buffer);
uint16_t RingBuffer_availableDataAmount(RingBuffer *buffer);
uint16_t RingBuffer_avalailableSpaceAmount(RingBuffer *buffer);


#define RingBuffer_available_data(B) (((B)->end + (B)->length - (B)->start) % (B)->length)
#define RingBuffer_available_space(B) ((B)->length - 1 - RingBuffer_available_data((B)))
#define RingBuffer_full(B) (RingBuffer_available_data((B)) - ((B)->length - 1) == 0)
#define RingBuffer_empty(B) (RingBuffer_available_data((B)) == 0)
#define RingBuffer_puts(B, D) RingBuffer_write((B), bdata((D)), blength((D)))
#define RingBuffer_get_all(B) RingBuffer_gets((B), RingBuffer_available_data((B)))
#define RingBuffer_starts_at(B) ((B)->buffer + (B)->start)
#define RingBuffer_ends_at(B) ((B)->buffer + (B)->end)
#define RingBuffer_commit_read(B, A) ((B)->start = ((B)->start + (A)) % (B)->length)
#define RingBuffer_commit_write(B, A) ((B)->end = ((B)->end + (A)) % (B)->length)

#endif
