#include "mbed.h"
#define DBGFLAG  1          //デバッグ出力フラグ 0：なし 1:結果のみ 2:入力含む 3〜:中間結果含む
#define BUFSIZE_BIT 8       //バッファサイズ(ビット数定義)
#define BUFSIZE (1<<BUFSIZE_BIT)    //バッファサイズ
#define BUFSIZE_MASK (BUFSIZE-1)    //バッファ演算マスク

#define IGNPULSE_WIDTH 3000         //出力イグニッションパルス幅

// ECU Simple1 alfa1  for mbed NXP LPC1768   2019/2/xx
//2019/2/27 周期ベースに書き換え。リングバッファのサイズ可変。
//          前のパルスのベクトル値を割り込み中に保存するようにして、関係各所で算出していたのを省略。

InterruptIn clankPulse(p14);
AnalogIn adc(p16);
Serial pc(USBTX, USBRX);
Timer t;                            //点火パルス発生タイミング
Timeout RearIgn_DelayTimer;         // 点火遅延タイマ、リア用
Timeout RearIgn_PulseWidthTimer;    // 点火パルス幅タイマ、リア用
 
 
DigitalOut IgnWait(LED2);           //点火パルス発生　遅延中
DigitalOut IgnPulse(LED3);          //点火パルス
DigitalOut ErrBIndicater(LED4);     //エラー（負論理）

DigitalOut PGTrigger(LED1);         //クランク回転パルス読み取り発生フラグ
//int PGTrigger;                    //クランク回転パルス読み取り発生フラグ排他利用

//パルスタイミング用バッファ定義
int clankPulseTime[BUFSIZE];        //クランク回転パルス リングバッファ（本体）
int clankPulseTimeBufSize = BUFSIZE;        //クランク回転パルス リングバッファのサイズ(本体の定義と一致させること)
int clankPulseTimeVector = 0;           //クランク回転パルス リングバッファの位置
int clankPulseTimeVectorBefore = BUFSIZE_MASK;  //クランク回転パルス リングバッファの位置（一つ前）

//初期化
void init() {
    pc.baud(115200);                //シリアル出力ボーレート(19200|38400|57600|115200)
    ErrBIndicater = 1 ;

    if ( DBGFLAG > 0 ) { 
        pc.printf("\n---- Nucleo LPC1768 ECU Simple1 test\r\n");
    }
    t.start();                  //タイマースタート
}


// ワンショットタイマ 割込みハンドラ(パルス幅設定)
void isrDlyOff(void) {
    IgnPulse = 0;                       // 点火パルス終了
}

// ワンショットタイマ 割込みハンドラ(点火開始まち)
void isrDlyOn(void) {
    IgnWait = 0;               // 点灯
    IgnPulse = 1;               // 点火パルス開始
    // 点火パルス幅 割込み設定、タイマスタート
    RearIgn_PulseWidthTimer.attach_us(isrDlyOff, IGNPULSE_WIDTH );   // 3ms
}

//点火処理(リア)
void RearIgn(int RearIgn_Delay) {
    // 点火パルス遅延 割込み設定、タイマスタート
    RearIgn_DelayTimer.attach_us(isrDlyOn, RearIgn_Delay);       // 2.3s
}

//クランク回転パルス　割込みハンドラ
void flip_rise(){             //risng edge interrupt
    PGTrigger = 0;
}

void flip_fall(){           //falling edge interrupt
    //リングバッファのベクトルをずらす。
    clankPulseTimeVectorBefore = clankPulseTimeVector;
    clankPulseTimeVector = (clankPulseTimeVector +1 ) & BUFSIZE_MASK;
    
    //キャプチャー時間列にシフト代入して、フラグ立てる。
    clankPulseTime[ clankPulseTimeVector ] = t.read_us() ;

    PGTrigger = 1;
            
    if ( DBGFLAG > 1 ) {
        pc.printf("\r\n" );
        pc.printf("flip_fall: clankPulseTime: %d [us] (scale X%d) ", clankPulseTime[ clankPulseTimeVector ] );
        pc.printf("address@%d", clankPulseTimeVector  );
        pc.printf("\r\n" );
    }

}


//クランク回転周期[us]読み取り
int readCrankCycle(){
    int timeCycle;  //周期[us]

//エンジン回転周期計算
    timeCycle = ( clankPulseTime[ clankPulseTimeVector ] - clankPulseTime[ clankPulseTimeVectorBefore ]);

    if ( timeCycle <= 0 ){
        timeCycle = -1;
    }

    if ( DBGFLAG > 2 ) {
        float timeCycle_sec = (float) timeCycle /1000000;
        float freqency = (float) 1 / timeCycle * 1000000;
        int engineRpm = (int) (  (float) 60 / timeCycle * 1000000 );

        pc.printf("readCrankCycle: ");
        pc.printf(" timeCycle %d[us] ", timeCycle , freqency );
        pc.printf("= after: %d ", clankPulseTime[ clankPulseTimeVector ] );
        pc.printf("- before: %d ", clankPulseTime[ clankPulseTimeVectorBefore ] );
        pc.printf("\r\n" );
        pc.printf("readCrankCycle:");
        pc.printf(" cycle: %f[sec] / engineRpm: %d", timeCycle_sec , engineRpm );
        pc.printf("\r\n" );
        pc.printf(" freqency: %f[Hz] ", freqency );
        pc.printf("\r\n" );
    }
    return timeCycle;

}

int ADC()
{
    float meas_r;
    int meas_int;
    
    meas_r = adc.read(); // Read the analog input value (value from 0.0 to 1.0 = full ADC conversion range)

    meas_int = (int)( (float)(meas_r * 100 ) + (float)0.49999 );

    // Display values
    if ( DBGFLAG > 1 ) {
        float meas_v;
        meas_v = meas_r * 3300; // Converts value in the 0V-3.3V range
        pc.printf("ADC: measure = %f = %.0f mV", meas_r, meas_v );
        pc.printf(" Return value = %d ", meas_int );
        pc.printf("\r\n" );
    }

    return meas_int;
}


//PCへログ出力　デバッグ用
void log2pc( int throttleOpenPercent ,int timeCycle ,int RearIgn_Delay) {
//エンジン回転数計算
    int engineRpm = (int) (  (float) 60 / timeCycle * 1000000 );

    pc.printf("Accell:%3d[%%]", throttleOpenPercent );
    pc.printf(" %5drpm " ,engineRpm );
    pc.printf(" %6.3f[ms] " ,(float)0.001 * timeCycle );
    pc.printf(" @vector%3d ", clankPulseTimeVector );
    pc.printf(" delay:%3d[ms] ", (float)0.001 * RearIgn_Delay );

    pc.printf("\r\n" );

//  ログ出力完了時間の記録
    int logTime = t.read_us() ;

//  ログ出力完了前にエンジンパルスが来たらエラー
    if ( logTime > clankPulseTime[clankPulseTimeVector]  ){
        //正常
    } else {
        ErrBIndicater = 0;
        pc.printf("\r\n" );
        pc.printf(" Log exe. time OVER %f" ,(float)( logTime - clankPulseTime[clankPulseTimeVector] )/1000000 );
        pc.printf(" = engine:%f - log:%f[s] ", (float)logTime/1000000, (float)(clankPulseTime[clankPulseTimeVector])/1000000 );
        pc.printf("\r\n" );
    }
// ログが歯抜けになったらエラー
    
}

int main() {
    int throttleOpenPercent[8]; //スロットル開度列
    int engineCycle;                //エンジン回転数/秒
    
    //clankPulse.rise(&flip_rise);          //回転パルス割り込み設定
    clankPulse.fall(&flip_fall);        //回転パルス割り込み設定
    init();                         //初期化

    while(1){
    //クランク回転パルス入力の立ち上がりでPGTriggerが降り、立ち下がりでフラグが１に立つので、
    //処理を開始する。
        if ( PGTrigger == 1 ) {
            engineCycle = readCrankCycle();
            
            //アクセル読み込み
            throttleOpenPercent[0] = ADC();

            //マップ読み込み
            int RearIgn_Delay = (float) engineCycle/200 * (101-throttleOpenPercent[0]) ;
            //点火出力
            IgnWait = 1;               // 点灯
            RearIgn( RearIgn_Delay);

            //デバッグ出力
            if ( DBGFLAG > 0 ) { 
                log2pc(throttleOpenPercent[0],engineCycle,RearIgn_Delay); 
            }
            PGTrigger = 0;          
        }

//  wait(0.01);
    }
}
