/* mbed MCP4261 Library, for driving the 7/8-Bit Single/Dual SPI Digital POT with Non-Volatile Memory
 * Copyright (c) 2015, Created by Steen Joergensen (stjo2809)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#include "mbed.h"

#ifndef MBED_MCP4261_H
#define MBED_MCP4261_H

//=============================================================================
// All The Addresses
//=============================================================================

#define TCON_ADDR       0x04       // Controls the state of each resistor network terminal connection.   
#define STATUS_ADDR     0x05       // Status (STATUS) Register, This register contains 5 status bits. WiperLock bits, Shutdown bit, Write Protect bit, EEPROM write cycle.
#define VW0_ADDR        0x00       // Volatile Wiper 0
#define VW1_ADDR        0x01       // Volatile Wiper 1
#define NVW0_ADDR       0x02       // Non Volatile Wiper 0
#define NVW1_ADDR       0x03       // Non Volatile Wiper 1

// DATA EEPROM locations has the address from 0x06 to 0x0F 

//=============================================================================
// Declaration of variables & custom #defines
//=============================================================================

#define CB_WRITE        0x00       // Device commad bit for WRITE
#define CB_INCR         0x01       // Device commad bit for INCREMENT
#define CB_DECR         0x02       // Device commad bit for DECREMENT
#define CB_READ         0x03       // Device commad bit for READ

//=============================================================================
// Functions Declaration
//=============================================================================

/** Interface to the 7/8-Bit Single/Dual SPI Digital POT with Non-Volatile Memory
 *
  *  Using the driver:
 *   - remenber to setup SPI in main routine or use pins instance.
 *
 *  Defaults in this driver on start up:
 *   - as default is HARDWARE WRITE PROTECT PIN "Off".
 *   - as default is HARDWARE SHUTDOWN PIN  "Off".
 *
 */
class MCP4261 {
public:
    /** Create an instance of the MCP4261 connected via specfied SPI instance.
     *
     * @param spi The mbed SPI instance (make in main routine)
     * @param nWP The Hardware Write Protect Control pin.
     * @param nSHDN The Shutdown pin.
     * @param nCs The SPI chip select pin.
     */
    MCP4261(SPI& spi, PinName nCs, PinName nWP, PinName nSHDN);
    
    /** Create an instance of the MCP4261 connected via specfied SPI instance (use wthen SHDN is linked to Global reset).
     *
     * @param spi The mbed SPI instance (make in main routine)
     * @param nWP The Hardware Write Protect Control pin.
     * @param nCs The SPI chip select pin.
     */
    MCP4261(SPI& spi, PinName nCs, PinName nWP);
    

    /** Read an Address.
     *
     * @param address The selected register to read from.
     * @return The 16 bits read.
     */
    int read(char address);

    /** Write to Address.
     *
     * @param address The selected register to write to.
     * @param data The 16 bits to write to the register
     */
    void write(char address, int data);
    
    /** Increment wiper.
     *
     * @param number The selected wiper to increment.
     */
    void inc(bool number);
    
    /** Decrement wiper.
     *
     * @param number The selected wiper to decrement.
     */
    void dec(bool number);
    
    /** Read the Status register.
     *
     * @return The 16 bits read.
     */
    int status();
    
    /** Read the tcon register.
     *
     * @return The 16 bits read.
     */
    int tcon();
    
    /** write to tcon register.
     *
     * @param data The 16 bits to write to the register
     */
    void tcon(int data);
    
    /** Read the Volatile Wiper.
     *
     * @param number The wiper number = '0' or '1'
     * @return The 16 bits read.
     */
    int wiper(bool number);
    
    /** write to Volatile Wiper.
     *
     * @param number The wiper number = '0' or '1'
     * @param data The 16 bits to write to the register
     */
    void wiper(bool number, int data);
    
    /** Read the non-volatile wiper (Power On Reset start value).
     *
     * @param number The wiper number = '0' or '1'
     * @return The 16 bits read.
     */
    int nvwiper(bool number);
    
    /** write to non-volatile wiper (Power On Reset start value).
     *
     * @param number The wiper number = '0' or '1'
     * @param data The 16 bits to write to the register
     */
    void nvwiper(bool number, int data);
    
    /** HARDWARE SHUTDOWN PIN (SHDN)
     *
     * @param act SHDN is Active = true and Inactive = false
     */
    void shdn(bool act);
    
    /** HARDWARE WRITE PROTECT PIN (WP)
     *
     * @param act WP is Active = true and Inactive = false
     */
    void wp(bool act);
  

private:
    SPI& _spi;
    DigitalOut _nCs;
    DigitalOut _nWP;
    DigitalOut _nSHDN;
    
        
    char _command_byte;
    
    char _make_command_byte(int com, char address, int data);
    int _read(char address);                          
    void _write(char address, int data);             

};

#endif