/*
    Copyright (C) 2017 romain reicher

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
    
    About 
    ---------------------------------------------------------------------

    Send sensors value to Ubidots.
    This example sends 3 variables to Ubidots.
        - STM32L476 VBAT/3 internal channel in mV
        - STM32L476 Internal Temperature Sensor in C
        - The status of onboard User Button (blue) on NucleoL476RG 
    Use NucleoL476RG with X-Nucleo-IDW01M1v2 wifi shield
 
    Important note: Some IDW01M1 wifi shield had resistor R21 mounted
                    which interfere with STLink/SWD programmer.
                    Please unmount R21 to fix it.
 
    romain reicher
    Date     :  20/09/2017
    Revision :  v0.1  
*/


#include "mbed.h"
#include "SpwfInterface.h"  
#include "TCPSocket.h"

/* Wifi Acces Point Settings */ 
#define AP_SSID         "mtn"            
#define AP_PASSWORD     "abcd1234"
#define UBIDOTS_SERVER  "things.ubidots.com"
#define UBIDOTS_PORT    80
#define UBIDOTS_TOKEN   "BBFF-tlXz9IxoP0VBBkqhdAKhxuPGrV7TLo"
#define UBIDOTS_DEVICE  "testrun"

/* Communication ressources */
SpwfSAInterface spwf(D8, D2, false);    
Serial pc(USBTX, USBRX);

/* Digital ressources */
DigitalOut myLed(LED1);
DigitalIn myButton(USER_BUTTON);

/* Analog ressources */
//AnalogIn adc_vbat(ADC_VBAT);    // VBAT / 3 internal to ADC channel
//AnalogIn adc_temp(ADC_TEMP);    // Internal Temp Sensor to ADC Channel

/* Global variables */
float temp = 27.0;//adc_temp.read() * 100;     // Converted in C
float batt = 3000; //adc_vbat.read() * 30000;   // Converted in mV 
float value = ubidots.get(DEVICE_LABEL_TO_RETRIEVE_VALUES_FROM, VARIABLE_LABEL_TO_RETRIEVE_VALUES_FROM); 
bool status = false;

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main() 
{
    /* Configure Serial baud rate */
    pc.baud(115200);
    
    /* Update status variable state depending USER_BUTTON state */
    if (myButton == 0)
        status = true;
    else
        status = false;
    
    TCPSocket socket(&spwf);
    char sendBuffer[256];
    char message[64];
    int err;
    
    /* ######################## WIFI CONNECTION ######################## */

    pc.printf("IDW01M1 NetworkSocketAPI TCP Client Ubidots\r\n");
    pc.printf("Connecting to AP\r\n");
    
    //* Connect to wifi acces point */    
    if(spwf.connect(AP_SSID, AP_PASSWORD, NSAPI_SECURITY_WPA2)) 
    {      
        pc.printf("Now connected\r\n");
    } 
    else
    {
        pc.printf("Error connecting to AP.\r\n");
        return -1;
    }   
    
    /* #################### GET CONNECTION INFOS ######################## */
    
    /* Get and print network connection parameters ip and mac adress */   
    const char *ip = spwf.get_ip_address();
    const char *mac = spwf.get_mac_address();    

    pc.printf("IP address is: %s\r\n", ip ? ip : "No IP");
    pc.printf("MAC address is: %s\r\n", mac ? mac : "No MAC");
    
    /* ##################### UBIDOATS SEND DATA ######################### */

    printf("Sending HTTP Data to Ubidots...\r\n");
 
    /* Open a socket , create a TCP connection to Ubidots */
    err = socket.connect(UBIDOTS_SERVER, UBIDOTS_PORT); 
    if (err!=0) 
    {
      pc.printf("\r\nCould not connect to Socket, err = %d!!\r\n", err); 
      return -1;
    } 
    else 
        pc.printf("\r\nconnected to host server\r\n"); 
    
    /* Construct content of HTTP command */
    sprintf(message, "{\"temperature\": %0.2f, \"battery\": %0.2f, \"status\": %d}", temp, batt, (int)status);
    printf("Content Length = %d\r\n", (int)strlen(message));
    
    /* Construct HTTP command to send */
    sprintf(sendBuffer, "POST /api/v1.6/devices/%s/?token=%s HTTP/1.1\r\nHost: things.ubidots.com\r\nContent-Type: application/json\r\nContent-Length: %d\r\n\r\n%s", UBIDOTS_DEVICE, UBIDOTS_TOKEN, (int)strlen(message),message); 
    pc.printf("HTTP command %s\r\n", sendBuffer);
    wait(2.0);
     
    /* Send http request to Ubidots */ 
    int scount = socket.send(sendBuffer, (int)strlen(sendBuffer));
    printf("sent %d [%.*s]\r\n", scount, strstr(sendBuffer, "\r\n") - sendBuffer, sendBuffer);

    /* Receive a simple http response and print out the response line */
    char respBuffer[64];
    int rcount = socket.recv(respBuffer, sizeof respBuffer);
    printf("recv %d [%.*s]\r\n", rcount, strstr(respBuffer, "\r\n") - respBuffer, respBuffer);

    /* Close the socket to return its memory and bring down the network interface */
    pc.printf("Close Socket\r\n");
    socket.close();
    
    /* Disconnect */
    pc.printf("Disconnect Wifi\r\n");
    spwf.disconnect();
    wait(1.0);
    pc.printf("Done\r\n");

    myLed = 0;
        
    while(1) 
    {
        myLed = !myLed;
        wait(1.0);
    }
}