/* Includes */
#include "mbed.h"
#include "XNucleoIKS01A2.h"
#include "MQTTTimer.h"
#include "CayenneMQTTClient.h"
#include "MQTTNetworkIDW01M1.h"
#include "SpwfInterface.h"

/* Instantiate the expansion board */
static XNucleoIKS01A2 *mems_expansion_board = XNucleoIKS01A2::instance(D14, D15, D4, D5);

/* Retrieve the composing elements of the expansion board */
static LSM303AGRMagSensor *magnetometer = mems_expansion_board->magnetometer;
static HTS221Sensor *hum_temp = mems_expansion_board->ht_sensor;
static LPS22HBSensor *press_temp = mems_expansion_board->pt_sensor;
static LSM6DSLSensor *acc_gyro = mems_expansion_board->acc_gyro;
static LSM303AGRAccSensor *accelerometer = mems_expansion_board->accelerometer;

// WiFi network info.
char* ssid = "iPhone";
char* wifiPassword = "abcd1234";

// Cayenne authentication info. This should be obtained from the Cayenne Dashboard.
char* username = "4f3fbcb0-3796-11e9-ad96-c15442ccb423";
char* password = "9e099f3d9aaedd7b76ca94044c6bb488c3999e3c";
char* clientID = "4fb9dca0-3f13-11e9-a6b5-e30ec853fbf2";

SpwfSAInterface interface(D8, D2); // TX, RX
MQTTNetwork<SpwfSAInterface> network(interface);
CayenneMQTT::MQTTClient<MQTTNetwork<SpwfSAInterface>, MQTTTimer> mqttClient(network, username, password, clientID);

/***********************************************************/
/* External sensor and output/actuators declared here      */
/***********************************************************/
DigitalOut led1(LED1);
//DigitalOut ledy(D1);

/* Helper function for printing floats & doubles */
static char *print_double(char* str, double v, int decimalDigits=2)
{
  int i = 1;
  int intPart, fractPart;
  int len;
  char *ptr;

  /* prepare decimal digits multiplicator */
  for (;decimalDigits!=0; i*=10, decimalDigits--);

  /* calculate integer & fractinal parts */
  intPart = (int)v;
  fractPart = (int)((v-(double)(int)v)*i);

  /* fill in integer part */
  sprintf(str, "%i.", intPart);

  /* prepare fill in of fractional part */
  len = strlen(str);
  ptr = &str[len];

  /* fill in leading fractional zeros */
  for (i/=10;i>1; i/=10, ptr++) {
    if (fractPart >= i) {
      break;
    }
    *ptr = '0';
  }

  /* fill in (rest of) fractional part */
  sprintf(ptr, "%i", fractPart);

  return str;
}

/**
* Print the message info.
* @param[in] message The message received from the Cayenne server.
*/
void outputMessage(CayenneMQTT::MessageData& message)
{
    switch (message.topic)  {
    case COMMAND_TOPIC:
        printf("topic=Command");
        break;
    case CONFIG_TOPIC:
        printf("topic=Config");
        break;
    default:
        printf("topic=%d", message.topic);
        break;
    }
    printf(" channel=%d", message.channel);
    if (message.clientID) {
        printf(" clientID=%s", message.clientID);
    }
    if (message.type) {
        printf(" type=%s", message.type);
    }
    for (size_t i = 0; i < message.valueCount; ++i) {
        if (message.getValue(i)) {
            printf(" value=%s", message.getValue(i));
        }
        if (message.getUnit(i)) {
            printf(" unit=%s", message.getUnit(i));
        }
    }
    if (message.id) {
        printf(" id=%s", message.id);
    }
    printf("\n");
}

/**
* Handle messages received from the Cayenne server.
* @param[in] message The message received from the Cayenne server.
*/
void messageArrived(CayenneMQTT::MessageData& message)
{
    int error = 0;
    // Add code to process the message. Here we just ouput the message data.
    outputMessage(message);

    if (message.topic == COMMAND_TOPIC) {
        switch(message.channel) {               /****************************************************/
        case 0:                                 /* the channel number for input                     */
            // Set the onboard LED state        /*                                                  */
            led1 = atoi(message.getValue());    /*can add other function after received input signal*/
            // Publish the updated LED state    /****************************************************/
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
        }
        
        // If this is a command message we publish a response. Here we are just sending a default 'OK' response.
        // An error response should be sent if there are issues processing the message.
        if ((error = mqttClient.publishResponse(message.id, NULL, message.clientID)) != CAYENNE_SUCCESS) {
            printf("Response failure, error: %d\n", error);
        }
    }
}

/**
* Connect to the Cayenne server.
* @return Returns CAYENNE_SUCCESS if the connection succeeds, or an error code otherwise.
*/
int connectClient(void)
{
    int error = 0;
    // Connect to the server.
    printf("Connecting to %s:%d\n", CAYENNE_DOMAIN, CAYENNE_PORT);
    while ((error = network.connect(CAYENNE_DOMAIN, CAYENNE_PORT)) != 0) {
        printf("TCP connect failed, error: %d\n", error);
        wait(2);
    }

    if ((error = mqttClient.connect()) != MQTT::SUCCESS) {
        printf("MQTT connect failed, error: %d\n", error);
        return error;
    }
    printf("Connected\n");

    // Subscribe to required topics.
    if ((error = mqttClient.subscribe(COMMAND_TOPIC, CAYENNE_ALL_CHANNELS)) != CAYENNE_SUCCESS) {
        printf("Subscription to Command topic failed, error: %d\n", error);
    }
    if ((error = mqttClient.subscribe(CONFIG_TOPIC, CAYENNE_ALL_CHANNELS)) != CAYENNE_SUCCESS) {
        printf("Subscription to Config topic failed, error:%d\n", error);
    }

    // Send device info. Here we just send some example values for the system info. These should be changed to use actual system data, or removed if not needed.
    mqttClient.publishData(SYS_VERSION_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, CAYENNE_VERSION);
    mqttClient.publishData(SYS_MODEL_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "mbedDevice");
    //mqttClient.publishData(SYS_CPU_MODEL_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "CPU Model");
    //mqttClient.publishData(SYS_CPU_SPEED_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "1000000000");

    return CAYENNE_SUCCESS;
}

/* Simple main function */
int main() {
    uint8_t id;
    float value1, value2, value3, value4;
    char buffer1[32], buffer2[32], buffer3[32], buffer4[32];
    int32_t axes[3];
  
    /* Enable all sensors */
    hum_temp->enable();
    press_temp->enable();
    magnetometer->enable();
    accelerometer->enable();
    acc_gyro->enable_x();
    acc_gyro->enable_g();
  
    printf("\r\n--- Starting new run ---\r\n");

    hum_temp->read_id(&id);
    printf("HTS221  humidity & temperature    = 0x%X\r\n", id);
    press_temp->read_id(&id);
    printf("LPS22HB  pressure & temperature   = 0x%X\r\n", id);
    magnetometer->read_id(&id);
    printf("LSM303AGR magnetometer            = 0x%X\r\n", id);
    accelerometer->read_id(&id);
    printf("LSM303AGR accelerometer           = 0x%X\r\n", id);
    acc_gyro->read_id(&id);
    printf("LSM6DSL accelerometer & gyroscope = 0x%X\r\n", id);
 
    // Initialize the network interface.
    printf("Initializing interface\n");
    interface.connect(ssid, wifiPassword, NSAPI_SECURITY_WPA2);

    // Set the default function that receives Cayenne messages.
    mqttClient.setDefaultMessageHandler(messageArrived);

    // Connect to Cayenne.
    if (connectClient() == CAYENNE_SUCCESS) {
        // Start the countdown timer for publishing data every 5 seconds. Change the timeout parameter to publish at a different interval.
        MQTTTimer timer(5000);

        while (true) {
                       
            // Yield to allow MQTT message processing.
            mqttClient.yield(1000);

            // Check that we are still connected, if not, reconnect.
            if (!network.connected() || !mqttClient.connected()) {
                network.disconnect();
                mqttClient.disconnect();
                printf("Reconnecting\n");
                while (connectClient() != CAYENNE_SUCCESS) {
                    wait(2);
                    printf("Reconnect failed, retrying\n");
                }
            }
        
            printf("\r\n");

            hum_temp->get_temperature(&value1);
            hum_temp->get_humidity(&value2);
            printf("HTS221: [temp] %7s C,   [hum] %s%%\r\n", print_double(buffer1, value1), print_double(buffer2, value2));
    
            press_temp->get_temperature(&value3);
            press_temp->get_pressure(&value4);
            printf("LPS22HB: [temp] %7s C, [press] %s mbar\r\n", print_double(buffer1, value3), print_double(buffer2, value4));

            printf("---\r\n");

            magnetometer->get_m_axes(axes);
            printf("LSM303AGR [mag/mgauss]:  %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);
    
            acc_gyro->get_x_axes(axes);
            printf("LSM6DSL [acc/mg]:      %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);

            accelerometer->get_x_axes(axes);
            printf("LSM303AGR [acc/mg]:  %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);
    
            acc_gyro->get_g_axes(axes);
            printf("LSM6DSL [gyro/mdps]:   %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);
    
            wait(1.5);

            
            // Publish some example data every few seconds. This should be changed to send your actual data to Cayenne.
            if (timer.expired()) {
                int error = 0;                                                                                                      /************************/
                if ((error = mqttClient.publishData(DATA_TOPIC, 1, TYPE_TEMPERATURE, UNIT_CELSIUS, value1)) != CAYENNE_SUCCESS) {   /* duplicate the if {}  */
                    printf("Publish temperature failed, error: %d\n", error);                                                       /* for more channel     */
                }                                                                                                                   /************************/
                if ((error = mqttClient.publishData(DATA_TOPIC, 2, TYPE_BAROMETRIC_PRESSURE, UNIT_PASCAL, value4)) != CAYENNE_SUCCESS) {
                    printf("Publish barometric pressure failed, error: %d\n", error);
                }
                if ((error = mqttClient.publishData(DATA_TOPIC, 3, TYPE_PROXIMITY, UNIT_METER, axes[1])) != CAYENNE_SUCCESS) {
                    printf("Publish proximity failed, error: %d\n", error);
                }
                // Restart the countdown timer for publishing data every 5 seconds. Change the timeout parameter to publish at a different interval.
                timer.countdown_ms(5000);
            }
        }    
    }
    else {
        printf("Connection failed, exiting\n");
    }

    if (mqttClient.connected())
        mqttClient.disconnect();
    if (network.connected())
        network.disconnect();
}
