#include "mbed.h"
#include "BLEDevice.h"
#include "IR.h"
#include "nRF51822n.h"

BLEDevice ble;

DigitalOut myled(LED1);
DigitalOut yourled(LED2);
nRF51822n nrf;

Serial py(USBTX, USBRX);

const static uint8_t beaconPayload[] = {
    0x00, 0x00, // Company identifier code (0x004C == Apple) - Should be all zero until we figure out this weird iOS bug
    0x02,       // ID
    0x15,       // length of the remaining payload
    0xE2, 0x0A, 0x39, 0xF4, 0x73, 0xF5, 0x4B, 0xC4, // UUID
    0xA1, 0x2F, 0x17, 0xD1, 0xAD, 0x07, 0xA9, 0x61,
    0x13, 0x37, // the major value to differenciate a location
    0xFA, 0xCE, // the minor value to differenciate a location
    0xC8        // 2's complement of the Tx power (-56dB)
};

extern GattService ir_service;
extern GattCharacteristic header, one, zero, ptrail, predata, code;

bool received_ir_transmission = false;
bool device_disconnected = true;

void onDataWritten(uint16_t handle)
{
    py.printf("Data written! %i\n", handle);
    for (int i = 0; i < ir_service.getCharacteristicCount(); i++) {
        GattCharacteristic* characteristic = ir_service.getCharacteristic(i);
        characteristic->getMaxLength();
        if (characteristic->getHandle() == handle) {
            uint16_t max_length = characteristic->getMaxLength();
            ble.readCharacteristicValue(handle, characteristic->getValuePtr(), &max_length);
            for (int i=0; i<max_length; i++) {
                py.printf("Got value: %d\n", characteristic->getValuePtr()[i]);
            }
            break;
        }
    }

    if (code.getHandle() == handle) {
        py.printf("Received complete instruction\n");
        received_ir_transmission = true;
    }
}

void disconnectionCallback(void)
{
    py.printf("Disconnected!\n");
    device_disconnected = true;
}

void connectionCallback(void)
{
    py.printf("Connected!\n");
    device_disconnected = false;
}

void onDataSent(uint16_t data)
{
    py.printf("onDataSent!\n");
}

int main()
{
    py.printf("Start of main\n");

    ble.init();
    ble.onConnection(connectionCallback);
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(onDataWritten);
    ble.onDataSent(onDataSent);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */

    ble.accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA,
                                     beaconPayload, sizeof(beaconPayload));

    ble.startAdvertising();

    ble.addService(ir_service);

    myled = 1;

    py.printf("Listening..\n");

    while (true) {
        ble.waitForEvent();
        if (received_ir_transmission && device_disconnected) {
            fireIRCode(header.getValuePtr(), one.getValuePtr(), zero.getValuePtr(), ptrail.getValuePtr(), predata.getValuePtr(), code.getValuePtr());
            received_ir_transmission = false;
            ble.startAdvertising();
        }
        myled = !myled;
    }
}