/**
@file GameFSM.h
@brief Header file of Game's Finite Stage Machine.
@brief This header file contains header file from Game.h, Clock.h, Menu.h, N5110.h and mbed.h.
@brief It also has functions of setting flag for four buttons.
@brief It shows stages of the whole programme and devided into 4 sectors and an initial stage with 
@brief regarding Game Mode sector, High Score sector, Game Screen sector, Your Score sector and Main Menu stage.
@brief Revision 1.0
@author Huynh Minh Tuan Le
@date April 2015
*/
#ifndef GAMEFSM_H
#define GAMEFSM_H

#include "Game.h"
#include "Clock.h"
#include "N5110.h"
#include "Menu.h"
#include "mbed.h"


Game game; ///<A game object declared from the Game class.
Clock countdown; ///<A countdown object declared from the Clock class. 
Menu menu; ///<A menu object declared from the Menu class.

/**
@namespace ButtonA
@brief Button A of interrupting service.
*/
InterruptIn ButtonA(p16);
/**
@namespace ButtonB
@brief Button B of interrupting service.
*/
InterruptIn ButtonB(p15);
/**
@namespace ButtonD
@brief Down button of interrupting service.
*/
InterruptIn ButtonD(p24);
/**
@namespace ButtonU
@brief Up button of interrupting service.
*/
InterruptIn ButtonU(p23);

int BuAFlag = 0; ///<A button A flag variable.
int BuBFlag = 0; ///<A button B flag variable.
int BuDFlag = 0; ///<A Down button flag variable.
int BuUFlag = 0; ///<A up button flag variable.

int stage = 0; ///<Stage variable for Finite Stage Machine.

Timer debounce; ///<A timer object for debouncing the buttons.

void BuAPress()
{
    debounce.start(); //Start timer for debounce object.
    if (debounce.read_ms()>200) {
        BuAFlag = 1; ///Set flag for button A
        debounce.stop(); //Stop timer
    }
}

void BuBPress()
{
    debounce.start(); //Start timer for debounce object.
    if (debounce.read_ms()>200) {
        BuBFlag = 1; ///Set flag for button B
        debounce.stop(); //Stop timer
    }
}

void BuDPress()
{
    debounce.start(); //Start timer for debounce object.
    if (debounce.read_ms()>200) {
        BuDFlag = 1; ///Set flag for button Down
        debounce.stop(); //Stop timer
    }
}

void BuUPress()
{
    debounce.start(); //Start timer for debounce object.
    if (debounce.read_ms()>200) {
        BuUFlag = 1; ///Set flag for button Up
        debounce.stop(); //Stop timer
    }
}

///Game FSM Class
class FSM
{
public:
    /**
    * The function contains FSM of the programme.
    */
    void proact();
    /**
    * The function that set all the buttons flag to 0. 
    */
    void resetButton();
};

void FSM::resetButton()
{
    BuDFlag = 0;
    BuUFlag = 0;
    BuAFlag = 0;
    BuBFlag = 0;
    lcd.clear();
}

void FSM::proact()
{
    switch(stage) {
        case 0: //Main stage 0: MAIN MENU - Cursor at Play line
            
            menu.main();
            lcd.drawCircle(23,27,2,1);
            if (BuDFlag) {
                resetButton();
                stage = 1;
            }
            if (BuAFlag) {
                resetButton();
                stage = 3;
            }
            
            break;

        case 1: //Main stage 1: MAIN MENU - Cursor at Score line
            
            menu.main();
            lcd.drawCircle(20,35,2,1);
            if (BuUFlag) {
                resetButton();
                stage = 0;
            }
            if (BuAFlag) {
                resetButton();
                stage = 2;
            }
            
            break;
            
//////////HIGH SCORE SECTOR//////////
        case 2: //(Main stage 2)Sub stage 0: HIGH SCORE sector - Cursor is at Easy line
            
            menu.highscore();
            lcd.drawCircle(23,19,2,1);
            if (BuDFlag) {
                resetButton();
                stage = 12; //Go to HIGH SCORE sector/Normal line
            }
            if (BuAFlag) {
                resetButton();
                stage = 14; //Go to HIGH SCORE sector/Hard high score
            }
            if (BuBFlag) {
                resetButton();
                stage = 1; //Back to MAIN MENU sector/Score line
            }
            
            break;

        case 12: //(Main stage 2)Sub stage 1: HIGH SCORE sector - Cursor is at Normal line
            
            menu.highscore();
            lcd.drawCircle(17,27,2,1);
            if (BuUFlag) {
                resetButton();
                stage = 2; //Go to HIGH SCORE sector/Easy line
            }
            if (BuDFlag) {
                resetButton();
                stage = 13; //Go to HIGH SCORE sector/Hard line
            }
            if (BuAFlag) {
                resetButton();
                stage = 15; //Go to HIGH SCORE sector/Normal high score
            }
            if (BuBFlag) {
                resetButton();
                stage = 1; //Back to MAIN MENU sector/Score line
            }
            
            break;

        case 13: //(Main stage 2)Sub stage 2: HIGH SCORE sector - Cursor is at Hard line
            
            menu.highscore();
            lcd.drawCircle(23,35,2,1);
            if (BuUFlag) {
                resetButton();
                stage = 12; //Go to HIGH SCORE sector/Normal line
            }
            if (BuAFlag) {
                resetButton();
                stage = 16; //Go to HIGH SCORE sector/Hard high score
            }
            if (BuBFlag) {
                resetButton();
                stage = 1; //Back to MAIN MENU sector/Score line
            }
            
            break;

        case 14: //(Main stage 2)Sub stage 3: HIGH SCORE sector - Easy mode high score
            
            lcd.printString("EASY MODE",16,0);
            lcd.printString("High score",14,2);
            lcd.printString("Your score",14,4);
            //Easy mode high score
            char HScore0[5];
            sprintf(HScore0,"%d",game.scoArr[0]);
            lcd.printString(HScore0,38,3);
            //Easy mode current score
            char HScore1[5];
            sprintf(HScore1,"%d",game.scoArr[1]);
            lcd.printString(HScore1,38,5);
            if (BuBFlag) {
                resetButton();
                stage = 2; //Back to HIGH SCORE sector/Easy line
            }
            
            break;

        case 15: //(Main stage 2)Sub stage 4: HIGH SCORE sector - Normal mode high score
            
            lcd.printString("NORMAL MODE",8,0);
            lcd.printString("High score",14,2);
            lcd.printString("Your score",14,4);
            //Normal mode high score
            char HScore2[5];
            sprintf(HScore2,"%d",game.scoArr[2]);
            lcd.printString(HScore2,38,3);
            //Normal mode current score
            char HScore3[5];
            sprintf(HScore3,"%d",game.scoArr[3]);
            lcd.printString(HScore3,38,5);
            if (BuBFlag) {
                resetButton();
                stage = 12; //Back to HIGH SCORE sector/Normal line
            }
            
            break;

        case 16: //(Main stage 2)Sub stage 5: HIGH SCORE sector - Hard mode high score
            
            lcd.printString("HARD MODE",16,0);
            lcd.printString("High score",14,2);
            lcd.printString("Your score",14,4);
            //Hard mode high score
            char HScore4[5];
            sprintf(HScore4,"%d",game.scoArr[4]);
            lcd.printString(HScore4,38,3);
            //Hard mode current score
            char HScore5[5];
            sprintf(HScore5,"%d",game.scoArr[5]);
            lcd.printString(HScore5,38,5);
            if (BuBFlag) {
                resetButton();
                stage = 13; //Back to HIGH SCORE sector/Hard line
            }
            
            break;
            
//////////RESET STAGE//////////
            //This case is a game condition setting case
            //Clock countdown was set to 60 seconds
            //Score was set to 0
        case 3: //Main stage 3: RESETING stage - Before entering GAME MODE menu
            game.reset();
            resetButton();
            stage = 4;
            break;
            
//////////GAME MODE SECTOR//////////
        case 4: //(Main stage 4)Sub stage 0: GAME MODE sector - Cursor at Easy line
            
            menu.gameset();
            lcd.drawCircle(23,19,2,1);
            if (BuAFlag) {
                resetButton();
                stage = 7;
            }
            if (BuBFlag) {
                resetButton();
                stage = 0;
            }
            if (BuDFlag) {
                resetButton();
                stage = 5; //Go to stage 8 of Game set menu
            }
            
            break;

        case 5: //(Main stage 4)Sub stage 1: GAME MODE sector - Cursor at Normal line
            
            menu.gameset();
            lcd.drawCircle(17,27,2,1);
            if (BuAFlag) {
                resetButton();
                stage = 8; //Go to stage 8: Normal mode game
            }
            if (BuBFlag) {
                resetButton();
                stage = 0; //Go back to stage 0: Main menu
            }
            if (BuUFlag) {
                resetButton();
                stage = 4; //Go to stage 4: Game mode menu
            }
            if (BuDFlag) {
                resetButton();
                stage = 6; //Go to stage 6: Game mode menu
            }
            
            break;

        case 6: //(Main stage 4)Sub stage 2: GAME MODE sector - Cursor at Hard line
            
            menu.gameset();
            lcd.drawCircle(23,35,2,1);
            if (BuAFlag) {
                resetButton();
                stage = 9;
            }
            if (BuBFlag) {
                resetButton();
                stage = 0;
            }
            if (BuUFlag) {
                resetButton();
                stage = 5;
            }
            
            break;
            
//////////GAME SCREEN SECTOR//////////
        case 7: //(Main stage 4)Sub stage 3: GAME SCREEN sector - Easy mode
            
            timer.attach(&timerExpired,1.0);
            gatimer.attach(&gatimerExpired,0.1); //Update time for Easy mode is 0.1s
            while(CClock > 0) {
                countdown.countDown();
                game.easyMode();
            }
            if (CClock == 0) {
                countdown.timeout();
            }
            resetButton();
            timer.detach();
            gatimer.detach();
            stage = 10; //Go to stage 10: Your score menu

            break;

        case 8: //(Main stage 4)Sub stage 4: GAME SCREEN sector - Normal mode
            
            timer.attach(&timerExpired,1.0);
            gatimer.attach(&gatimerExpired,0.2); //Update time for Normal mode is 0.2s
            while(CClock > 0) {
                countdown.countDown();
                game.norMode(); //Normal mode of game is activated
            }
            if (CClock == 0) {
                countdown.timeout();
            }
            resetButton();
            timer.detach();
            gatimer.detach();
            stage = 10; //Go to stage 10: Your score menu

            break;

        case 9: //(Main stage 4)Sub stage 5: GAME SCREEN sector - Hard mode
            
            timer.attach(&timerExpired,1.0);
            gatimer.attach(&gatimerExpired,0.4); //Update time for Hard mode is 0.4s
            while(CClock > 0) {
                countdown.countDown();
                game.hardMode(); //Hard mode of the game is activated
            }
            if (CClock == 0) {
                countdown.timeout();
            }
            resetButton();
            timer.detach();
            gatimer.detach();
            stage = 10; //Go to stage 10: Your score menu

            break;
            
//////////YOUR SCORE SECTOR//////////
        case 10: //Main stage 5: YOUR SCORE sector - Cursor at Main menu line
            
            menu.yourscore();
            lcd.drawCircle(8,27,2,1);
            char YScore[14];
            sprintf(YScore,"%d",game.score);
            lcd.printString(YScore,37,2);
            if (BuDFlag) {
                resetButton();
                stage = 11;
            }
            if (BuAFlag) {
                resetButton();
                stage = 0;
            }
            
            break;

        case 11: //Main stage 6: YOUR SCORE sector - Cursor at Reset line
            
            menu.yourscore();
            lcd.drawCircle(8,35,2,1);
            lcd.printString(YScore,37,2);
            if (BuUFlag) {
                resetButton();
                stage = 10;
            }
            if (BuAFlag) {
                resetButton();
                stage = 3;
            }
            
            break;
    }
    wait(0.1);
}
#endif