#include "i2c_api.h"
#include "wait_api.h"

/* Driver Object Definition */       
struct sDS2482_t
{    
    uint32_t u32Flags;
    uint8_t u8Addr;
    
    uint8_t u8RomNr[8];
    int16_t i16LastDiscrepancy;
    int16_t i16LastFamilyDiscrepancy;
    int16_t i16LastDeviceFlag;
    uint8_t u8CRC8;
} ;

/* DS2482 Low-Level-Interface */
int8_t  i8DS2482Reset               (struct sDS2482_t *dev);
int16_t i16DS2482GetStatus          (struct sDS2482_t *dev);
int8_t  i8DS2482SetControlBits      (struct sDS2482_t *dev, uint8_t u8Flags);
// TODO: extract basic write/read function from higher-level functions below

/* 1Wire Interface Functions */
int8_t  i8DS2482_OWWait             (struct sDS2482_t *dev);
int8_t  i8DS2482_OWReset            (struct sDS2482_t *dev);
int8_t  i8DS2482_OWTouchBit         (struct sDS2482_t *dev, uint8_t sendbit);
int8_t  i8DS2482_OWReadBit          (struct sDS2482_t *dev);
int8_t  i8DS2482_OWWriteBit         (struct sDS2482_t *dev, uint8_t sendbit);
int8_t  i8DS2482_OWWriteByte        (struct sDS2482_t *dev, uint8_t sendbyte);
int16_t i16DS2482_OWReadByte        (struct sDS2482_t *dev);
int8_t  i8DS2482_OWBlock            (struct sDS2482_t *dev, uint8_t *tran_buf, uint8_t tran_len);
int16_t i16DS2482_OWTouchByte       (struct sDS2482_t *dev, uint8_t sendbyte);
int16_t i16DS2482_OWFirst           (struct sDS2482_t *dev);
int16_t i16DS2482_OWNext            (struct sDS2482_t *dev);
int16_t i16DS2482_OWSearch          (struct sDS2482_t *dev);
int16_t i16DS2482_search_triplet    (struct sDS2482_t *dev, int search_direction);
int8_t  i8DS2482_OWSelectDevice     (struct sDS2482_t *dev, uint8_t *u8SN);
int8_t  i8DS2482_OWCheckDeviceReady (struct sDS2482_t *dev);

/* DS1820-Specific High-Level-Functions */
int8_t  i8DS2482_OWStartAllDS1820   (struct sDS2482_t *dev, uint8_t u8WaitForCompletion);
int16_t i16DS2482_OWReadDS1820      (struct sDS2482_t *dev, uint8_t *u8SN, uint8_t u8ManualStart);
int8_t  i8DS2482_OWReadDS1820Precise(struct sDS2482_t *dev, uint8_t *u8SN, uint8_t u8ManualStart, int16_t *i16Temperature);

/* Helper Function */
uint8_t calc_crc8 (uint8_t *data_in, int number_of_bytes_to_read);

/* =============== USAGE EXAMPLE =============== 
#include "mbed.h"
#include "ds2482.h"

#define MAX_TEMP_SENSORS        16
#define CONNECTED_DS2482_HUBS   2

struct sDS1820_t
{    
    struct sDS2482_t *hub;
    uint8_t u8RomNr[8];
};

struct sDS1820_t sDS1820[MAX_TEMP_SENSORS];
struct sDS2482_t sDS2482[CONNECTED_DS2482_HUBS];

Serial console(USBTX, USBRX);
I2C i2c (p9, p10);

int8_t i8SetupTempSensors(void)
{
    int x=0;    
    
    sDS2482[0].u8Addr = DS2482_ADDR1;     
    sDS2482[1].u8Addr = DS2482_ADDR2;
    
    for(int loop=0; loop<2; loop++)
    {   
        int8_t i8Tmp = i8DS2482Reset(&sDS2482[loop]);
        if(i8Tmp)
            return i8Tmp;
        
        i8Tmp = i8DS2482SetControlBits(&sDS2482[loop], APU | SPU );
        if(i8Tmp)
            return i8Tmp;
        
        i8Tmp = i8DS2482_OWReset(&sDS2482[loop]);        
        if(i8Tmp) 
            return i8Tmp;
            
        while(i16DS2482_OWSearch(&sDS2482[loop]) > 0)
        {            
            sDS1820[x].hub = &sDS2482[loop];
            for(int z=0; z<8; z++)
                sDS1820[x].u8RomNr[z] = sDS2482[loop].u8RomNr[z];                        
            x++;
        }
    }  
    return x;
}

int main(void)
{
    uint8_t u8SensorCount;
    
    mbed_i2c = &i2c; 
    
    console.baud(115200);    
    
    int8_t i8Ret = i8SetupTempSensors();    
    
    if(i8Ret < 0)
    {
        console.printf("Error -i8Ret\n");    
        while(1);       // error occured
    }
    
    u8SensorCount = i8Ret;
    
    while(1)
    {
        // Start Temperature Conversion on all DS1820
        for(uint8_t loop = 0; loop < CONNECTED_DS2482_HUBS; loop++)
        {            
            i8Ret = i8DS2482_OWStartAllDS1820(&sDS2482[loop], 0);
                if(i8Ret) 
                {
                    console.printf("Error %i\n", -i8Ret);
                    while(1);   // error!            
                }
        }
        
        // Wait until all DS1820 have completed the conversion
        for(uint8_t loop = 0; loop < CONNECTED_DS2482_HUBS; loop++)        
            while(!i8DS2482_OWCheckDeviceReady(&sDS2482[loop]));                        
        
        // Get temperature values and display them
        for(uint8_t z=0; z<u8SensorCount; z++)
        {
            int16_t i16Tmp = i16DS2482_OWReadDS1820(sDS1820[z].hub, sDS1820[z].u8RomNr, 0);
            if(i16Tmp < 0)    
            {                
                console.printf("Error %i\n", -i16Tmp);
                while(1);           // error                    
            }
            else
            {
                uint8_t u8Tmp = (i16Tmp-109)/2;
                uint8_t u8Tmp2;
                if((int16_t)u8Tmp*2+109 != i16Tmp)
                    u8Tmp2=5;
                else
                    u8Tmp2=0;
                console.printf("[%02i] %02i", z+1, u8Tmp);         
                console.printf(",%iC | ", u8Tmp2);
            }
            if((z+1)%8==0)
                console.printf("\n");
        }                                        
    }
}
*/

/* HW Setup */
#define DS2482_ADDR1    0x30
#define DS2482_ADDR2    0x32

#define POLL_LIMIT      50

#define ACK     1
#define NAK     0

/* DS2482 Status Register Bit Definitions */
#define STATUS_1WB      (1<<0)
#define STATUS_PPD      (1<<1)
#define STATUS_SD       (1<<2)
#define STATUS_LL       (1<<3)
#define STATUS_RST      (1<<4)
#define STATUS_SBR      (1<<5)
#define STATUS_TSB      (1<<6)
#define STATUS_DIR      (1<<7)

/* DS2482 Config Register Definitions */
#define APU     0x1             // active pullup
#define SPU     0x4             // strong pullup
#define OWS     0x8             // 1wire speed

/* Flag Definition */
#define FLAG_SHORT      0x00000001

/* Error Codes */
#define DS2482_ERR_NOPRESENCE       2
#define DS2482_ERR_TIMEOUT          3
#define DS2482_ERR_I2CWRITE         4
#define DS2482_ERR_I2CREAD          5
#define DS2482_ERR_CONFIGMISSMATCH  6
#define DS2482_ERR_CHECKSUM         7

/* DS2482 Commands */
#define DRST    0xF0
#define SRP     0xE1
#define WCFG    0xD2
#define OWRS    0xB4
#define OWSB    0x87
#define OWWB    0xA5
#define OWRB    0x96
#define OWT     0x78

/* 1Wire CRC Definitions */
#define CRC8INIT    0x00
#define CRC8POLY    0x18  

