/**
  *  \brief Data location data holder
  **/
// Example Usage
// Looping: for(int i=0;i<DH::Locs().getI(LHType_locs,LHIType_size);i++){DataLocation thisData=DH::Locs().getC(LHType_locs,i);}
// Appending: DH::Locs().add(LHType_locs,DataLocation(4.0f,5.0f,6.0f));
// ResetStart: DH::Locs().inc(LHType_locs,0,true);
// ResetEnd: DH::Locs().inc(LHType_locs,-1,true);
// Increment: DH::Locs().inc(LHType_locs);
// GetCurrent: DH::Locs().getC(LHType_locs,DH::Locs().getI(LHType_locs));
#ifndef _DATA_LOCATION_H_
#define _DATA_LOCATION_H_
#define MAXNUMLOCS 64

#include "usb.h"

typedef struct DataLocS{
    double lat;
    double lon;
    double alt;
    double timestamp;
    double heading;
    double tilt;
}DataLoc;

// Storage of data location
class DataLocation{
private:
    // Current value of lat lon and alt
   DataLoc loc;
public:
    DataLocation(){}
    DataLocation(double latA,double lonA,double altA,double timestampA=0,double headingA=0){
        loc.lat=latA;
        loc.lon=lonA;
        loc.alt=altA;
        loc.timestamp=timestampA;
        loc.heading=headingA;
    }
    double& getLat(){return loc.lat;}
    double& getLon(){return loc.lon;}
    double& getAlt(){return loc.alt;}
    double& getTime(){return loc.timestamp;}
    double& getHeading(){return loc.heading;}
    double& getTilt(){return loc.tilt;}
    DataLoc& getLoc(){return loc;}
};

/// \brief Location holder type
enum LHType{
    LHType_locs=0,
    LHType_targ,
    LHType_base
};

/// \brief Location holder index type
enum LHIType{
    LHIType_head=0,
    LHIType_size
};

// Singleton location holder
class LocHolder{
private:
    // Actual Locations (absolute)
    DataLocation locs[MAXNUMLOCS];
    // Target Locations (relative to base station -> base station is at 0,0,0)
    DataLocation targ[MAXNUMLOCS];
    // Base Station Locations (absolute)
    DataLocation base[MAXNUMLOCS];
    // Index of the head of the circular buffers
    unsigned int headLocs,headTarg,headBase;
    // Number of locations
    unsigned int sizeLocs,sizeTarg,sizeBase;
public:
    /// \brief Default constructor
    LocHolder():headLocs(0),headTarg(0),headBase(0),sizeLocs(0),sizeTarg(0),sizeBase(0){}
    
    /// \brief Get locations type
    DataLocation* get(LHType type=LHType_locs);
    
    /// \brief Get Current value
    // Set the offset to a positive number to start from beg
    // Set the offset to a negative number to start from end 
    DataLocation& getC(LHType type=LHType_locs,int offset=0);
    
    /// \brief Get Index
    // Set type to head to find get the index of the current incremeter.
    // Set type to size to find the size
    unsigned int& getI(LHType type=LHType_locs,LHIType indexType=LHIType_head);
    
    /// \brief Fix an index that might be out of bounds;
    unsigned int getRealIndex(LHType type,int index,int offset=0,bool useSize=true);
    
    /// \brief Increment index (loop on overflow)
    void inc(LHType type=LHType_locs,int amount=1,bool abs=false);
    
    /// \brief Append a location to the end.
    void add(LHType type,DataLocation newLoc);

};
class DH{
public:
    static LocHolder& Locs();
};
#endif