
#ifndef MBED_SharpLCD_H
#define MBED_SharpLCD_H

#include "mbed.h"
#include <stdarg.h>
#include "BurstSPI.h"

#define incx() x++, dxt += d2xt, t += dxt
#define incy() y--, dyt += d2yt, t += dyt

/// Set the display geometry depending on the resolution of the display used
/// common types are 96x96, 128x128, 400x240
/** MemoryLCD width in pixels */
#define DISPLAY_WIDTH				(128)
/** MemoryLCD height in pixels */
#define DISPLAY_HEIGHT				(128)

/** Maximum length of a printf to the display */
#define MAX_PRINTF_CHARS			40
/** Data type for storing buffer the pixel buffer */
#define	DISPLAY_BUFFER_TYPE			uint8_t
#define DISPLAY_BUFFER_TYPE_SIZE	(sizeof(DISPLAY_BUFFER_TYPE) * 8)
#define DISPLAY_BUFFER_ELEMENTS 	((DISPLAY_WIDTH*DISPLAY_HEIGHT)/DISPLAY_BUFFER_TYPE_SIZE)

/** Color definitions */
#define White			0xFFFF
#define Black			0x0000
#define foreground		White
#define background		Black

/** Sharp Memory LCD.
 *  Supports 96x96, 128x128, 400x240 displays.
 *  Uses a Frame Buffer in RAM depending on resolution of display as follows:
 *  96 Display - 1.7k, 128 Display - 2.6k, 400 Display - 12.4k
 *
 * Example:
 * @code
#include "mbed.h"
#include "SharpLCD.h"

//KL25
SharpLCD display(PTC6, NC, PTC5, PTC3, PTC4, NC); //mosi, miso(not used), sck, cs, enable, extcom

int main()
{        
    display.enableDisplay();    

    while(1)
    {
       display.clearImmediate();
        // draw ellipse (To draw circle, set the last two axis the same)
        for (int j=0; j<60; j++) {
            display.ellipse(64,64,j,60,Black);
            display.update();
            //wait(.01);
            display.ellipse(64,64,j,50,White); // offset last axis here gives interesting effect!
            display.locate(5,6); 
            display.printf("Sharp");
            display.locate(2,8); 
            display.printf("Draw Ellipse");
            display.locate(4,10); 
            display.printf("MemoryLCD"); 
            display.update();
        }
        // draw rectangle
        for (int j=0; j<128; j++) {
            display.rect(0,0,j,j,Black);
            display.update();
            //wait(.01);
            display.rect(0,0,j,j,White); 
            display.locate(5,6); 
            display.printf("Sharp");
            display.locate(1,8); 
            display.printf("Draw Rectangle");
            display.locate(4,10); 
            display.printf("MemoryLCD"); 
            display.update();
        }
         for (int j=60; j>0; j--) {
            display.circle(64,64,j,Black);
            display.update();
            wait(.01);
            display.circle(64,64,j,White); 
            display.locate(5,6); 
            display.printf("Sharp");
            display.locate(2,8); 
            display.printf("Draw Circle");
            display.locate(4,10); 
            display.printf("MemoryLCD"); 
            display.update();
        }   
   	}            
}   
 * @endcode
 */


class SharpLCD {
public:
	 
	SharpLCD(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName enable, PinName extcom);
	
    /** 
    * Sets external font usage, eg. dispaly.set_font(Arial12x12);
    * This uses pixel positioning.
    * display.set_font(NULL); returns to internal default font.
    * void set_font(const unsigned char * f);    
    */
    void set_font(const unsigned char * f);    
    /** 
    * Sets position of the next character or string print.
    * External font, set pixel x(column),y(row) position.
    * internal(default) font, set character column and row position
    */
    void locate(int column, int row);
    /**
    * printf (from Stream).
    * Full printf funtionality only available with internal(default) font.
    * CR and LF have no effect with external fonts.
    */
    void printf(const char* format, ...);
    /**
    * Enables (power) display.
    */
    void enableDisplay(void);
    /**
    * Dissables (power) display.
    */   
    void disableDisplay(void);
    /**
	 * Set a pixel in the pixel buffer.
     *
     * @param x      X-Position.
     * @param y      Y-Position.
     * @param colour Colour to set pixel to. White or Black.
	 */     
    void pixel(int x, int y, int colour);
    /**
	 * Draw rectangle one pixel wide (wire frame) in the pixel buffer.
     *
     * @param x0      X-Start position.
     * @param y0      Y-Start position.
     * @param x1      X-End position.
     * @param y1      Y-End position.
     * @param colour Colour to set pixel to. White or Black.
	 */                
    void rect(int x0, int y0, int x1, int y1, int colour);
    /**
	 * Draw filled rectangle in the pixel buffer.
     *
     * @param x0      X-Start position.
     * @param y0      Y-Start position.
     * @param w       Width.
     * @param h       Height.
     * @param colour Colour to fill rectangle to. White or Black.
	 */        
    void fillrect(int x0, int y0, int w, int h, int colour);
    /**
	 * Draw Circle one pixel wide in the pixel buffer.
     *
     * @param x0      X-Centre position.
     * @param y0      Y-Centre position.
     * @param r       Radius.
     * @param colour Colour to set pixel to. White or Black.
	 */        
    void circle(int x, int y, int r, int colour);
    /**
	 * Draw Ellipse or Circle one pixel wide in the pixel buffer.
     * Set v & h to same values to produce a circle.
      
     * @param xc      X-Centre position.
     * @param yc      Y-Centre position.
     * @param v       Vertical radius.
     * @param h       Horizontal radius.
     * @param colour Colour to set pixel to. White or Black.
	 */        
    void ellipse(int xc, int yc, int v, int h, unsigned int colour);
    /**
	 * Draw Filled Ellipse or Circle in the pixel buffer.
     * Set v & h to same values to produce a circle.
      
     * @param xc      X-Centre position.
     * @param yc      Y-Centre position.
     * @param v       Vertical radius.
     * @param h       Horizontal radius.
     * @param colour Colour to set pixel to. White or Black.
	 */        
    void fillellipse(int xc, int yc, int v, int h, unsigned int colour);
    /**
	 * Draw a horizontal line one pixel wide in the pixel buffer.
     *
     * @param x0      X-Start position.
     * @param x1      X-End position.
     * @param y       Y-position.
     * @param colour Colour to set pixel to. White or Black.
	 */     
    void hline(int x0, int x1, int y, int colour);
    /**
	 * Draw a Vertical line one pixel wide in the pixel buffer.
     *
     * @param x0      X-Position.
     * @param y0      Y-Start position.
     * @param y1      Y-End position.
     * @param colour Colour to set pixel to. White or Black.
	 */     
    void vline(int x0, int y0, int y1, int colour);
    /**
	 * Draw a Line in the pixel buffer.
     *
     * @param x0      X-Start position.
     * @param y0      Y-Start position.
     * @param x1      X-End position.
     * @param y1      Y-End position.
     * @param colour Colour to fill rectangle to. White or Black.
	 */        
    void line(int x0, int y0, int x1, int y1, int colour);
    /**
    * Toggles the EXTCOM connection to the display if used.
    * Call this function at 55 ~ 65 Hz to keep the display from losing contrast.
    */
    void toggle();
	/**
	 * Transfers pixel buffer to the display
	 */	
	void update();	
	/**
	 * Clear the display & set pixel buffer to zero.
	 */	
	void clearImmediate();	
	/**
     * Move bitmap into pixel buffer
     * 
     * @param bitmap      pointer to uint8 array containing horizontal pixel data
     * @param bmpWidth    width of the bitmap in pixels (must be byte multiple, eg 8,16,32,...)
     * @param bmpHeight   height of the bitmap in pixels (must be byte multiple, eg 8,16,32,...)
     * @param startX      starting position to apply bitmap in horizontal direction (0 = leftmost) (pixel resolution)
     * @param startY      starting position to apply bitmap in vertical direction (0 = topmost) (pixel resolution)
     */
   void showBMP(const uint8_t* bitmap, const uint32_t bmpWidth, const uint32_t bmpHeight, const uint32_t startX, const uint32_t startY);
   /** Output a character at the given position
     *
     * @param column column where charater must be written
     * @param  row where character must be written
     * @param c the character to be written to the TextDisplay
     */
   void character(int column, int row, int c);
   
    /** return number if rows on TextDisplay
     * @result number of rows
     */
    int rows();
    /** return number if columns on TextDisplay
    * @result number of rows
    */
    int columns();   

private:
	void window(int x, int y, int w, int h);
    void putp(int colour);
	void blit(int x, int y, int w, int h, const int *colour);    
    void blitbit(int x, int y, int w, int h, const char* colour);	
protected:
	//SPI        spi; // standard SPI
	BurstSPI spi; // Faster SPI pixel buffer transfer
    DigitalOut chipSelect;
    DigitalOut Enable;
    DigitalOut ExtCom;

	uint8_t lcdPolarity;
	volatile uint32_t rowCount;	
	uint8_t cmd[2 + (DISPLAY_WIDTH / DISPLAY_BUFFER_TYPE_SIZE * sizeof(DISPLAY_BUFFER_TYPE))];
	volatile DISPLAY_BUFFER_TYPE pixelBuffer[DISPLAY_BUFFER_ELEMENTS]; // one full frame buffer
	volatile DISPLAY_BUFFER_TYPE RowState[DISPLAY_HEIGHT/DISPLAY_BUFFER_TYPE_SIZE]; // 1 bit per row to indicate row change status
		
	// transfers pixel buffer to display
	void display_write();	 
	
	 // pixel location
    short _x;
    short _y;
    
    // window location
    short _x1;
    short _x2;
    short _y1;
    short _y2;
   	
    int _putc(int value);
    int _getc();
    int width();
    int height(); 
    
    // external font functions
    const unsigned char* font;
    int externalfont;

    // character location
    int _column;
    int _row;
    unsigned int char_x;
    unsigned int char_y; 

};
#endif